/*
                   Wright 1901 Wind Tunnel Simulation
                        angle of attack input
                  special graphics for lift and drag balance
                    made from FoilSim II  - Airfoil  mode
                  Kiosk  Version  -- performs plots of output
   
                           A Java Applet
               to perform Kutta-Joukowski Airfoil analysis

                     Version 2.2b   - 5 Apr 04


                         Written by Tom Benson
                       NASA Glenn Research Center

>                              NOTICE
>This software is in the Public Domain.  It may be freely copied and used in
>non-commercial products, assuming proper credit to the author is given.  IT
>MAY NOT BE RESOLD.  If you want to use the software for commercial
>products, contact the author.
>No copyright is claimed in the United States under Title 17, U. S. Code.
>This software is provided "as is" without any warranty of any kind, either
>express, implied, or statutory, including, but not limited to, any warranty
>that the software will conform to specifications, any implied warranties of
>merchantability, fitness for a particular purpose, and freedom from
>infringement, and any warranty that the documentation will conform to the
>program, or any warranty that the software will be error free.
>In no event shall NASA be liable for any damages, including, but not
>limited to direct, indirect, special or consequential damages, arising out
>of, resulting from, or in any way connected with this software, whether or
>not based on warranty, contract, tort or otherwise, whether or not injury
>was sustained by persons or property or otherwise, and whether or not loss
>was sustained from, or arose out of the results of, or use of, the software
>or services provided hereunder.
 
  New test -
            * add all of the Wright airfoil models
            * add the drag balance
            * calibrate against Wright data
            * smooth transitions on graphics
            * change display for kiosk
            * add plotter from VuShock
            * output the drag/lift ratio for the drag balance
            * correct problem in data reduction for lift coefficient
              cleanup
 
                                           TJB  5 Apr 04

*/

import java.awt.*;
import java.lang.Math ;

public class Foil extends java.applet.Applet {
 
   static double convdr = 3.1415926/180. ;
   static double pid2 = 3.1415926/2.0 ;
   static double rval,ycval,xcval,gamval,alfval,thkval,camval,chrd,clift,cdrag ;
   static double thkinpt,caminpt ;                 /* MODS 10 Sep 99 */
   static double leg,teg,lem,tem;
   static double usq,vsq,alt,altmax,area,armax,armin ;
   static double chord,span,aspr,arold,chrdold,spnold ; /* Mod 13 Jan 00 */
   static double q0,ps0,pt0,ts0,rho,rlhum,temf,presm ;
   static double lyg,lrg,lthg,lxgt,lygt,lrgt,lthgt;/* MOD 20 Jul */
   static double lxm,lym,lxmt,lymt,vxdir;/* MOD 20 Jul */
   static double deltb,xflow ;             /* MODS  20 Jul 99 */
   static double delx,delt,vfsd,spin,spindr,yoff ;
   static double vel,pres,lift,side,omega,radcrv,relsy,angr ;
   static double weight,aread,sped ;
   static double angout,glidang,angdraw,angdraw2 ;

   static double rg[][]  = new double[25][75] ; 
   static double thg[][] = new double[25][75] ; 
   static double xg[][]  = new double[25][75] ; 
   static double yg[][]  = new double[25][75] ; 
   static double xm[][]  = new double[25][75] ; 
   static double ym[][]  = new double[25][75] ; 
   static double plp[]   = new double[75] ;
   static double plv[]   = new double[75] ;

   int nptc,npt2,nlnc,nln2,rdflag,browflag,probflag,anflag;
   int foil,flflag,lunits,lftout,planet,plane ;
   int conflag,displ,dispp,antim,ancol;   /* MODS  2 APR 99 - 22 APR -27 JUL */
   int varflg ;
   int starter,stepper,balance,group,model,ysl,pegged,trans1,trans2 ;
       /* units data */
   static double vmn,almn,angmn,vmx,almx,angmx ;
   static double camn,thkmn,camx,thkmx ;
   static double chrdmn,spnmn,armn,chrdmx,spnmx,armx ;
   static double vconv,vmaxa,vmaxb ;
   static double pconv,pmax,pmin,lconv,fconv,fmax,fmaxb;
   int lflag,gflag,plscale,nond;
       /*  plot & probe data */
   static double fact,xpval,ypval,pbval,factp;
   static double prg,pthg,pxg,pyg,pxm,pym ;
   int pboflag,xt,yt,ntikx,ntiky,npt,xtp,ytp ;
   int lines,nord,nabs,ntr,nplts ;
   static double begx,endx,begy,endy ;
   static String labx,labxu,laby,labyu ;
   static double pltx[][]  = new double[6][50] ;
   static double plty[][]  = new double[6][50] ;
   static double xlegn[]  = new double[6];
   static double ylegn[]  = new double[6];
   static int ncurv[]  = new int[6] ;
   static int legnd[]  = new int[6] ;

   Solver solve ;
   Sq sq ;
   Lq lq ;
   CardLayout layin ;
   Image offImg1 ;
   Graphics off1Gg ;
   Image offImg2 ;
   Graphics off2Gg ;
   Image offImg3 ;
   Graphics off3Gg ;
   Image partimg ;

   public void init() {
     int i;
     Foil a = new Foil() ;
     solve = new Solver() ;

     offImg1 = createImage(this.size().width,
                      this.size().height) ;
     off1Gg = offImg1.getGraphics() ;
     offImg2 = createImage(this.size().width,
                      this.size().height) ;
     off2Gg = offImg2.getGraphics() ;
     offImg3 = createImage(this.size().width,
                      this.size().height) ;
     off3Gg = offImg3.getGraphics() ;

     setLayout(new GridLayout(1,2,5,5)) ;

     solve.setDefaults () ;
 
     sq = new Sq(this) ;
     lq = new Lq(this) ;

     add(lq) ;
     add(sq) ;

     solve.getFreeStream ();
     computeFlow () ;
     partimg = getImage(getCodeBase(),"tunnel.gif");
     lq.view.start() ;
     sq.start() ;
     lq.ob.inright.l.repaint() ;
  }
 
  public Insets insets() {
     return new Insets(5,5,5,5) ;
  }

  public void computeFlow() { 

     if (flflag == 1) {
         solve.getCirc ();                   /* get circulation */
         solve.genFlow () ;
         solve.getFreeStream () ;
     }

     solve.getProbe() ;
 
     loadOut() ;
     lq.ob.inright.l.repaint() ;
     sq.repaint() ;
  }

  public int filter0(double inumbr) {
        //  output only to .
       int number ;
       int intermed ;
 
       number = (int) (inumbr);
       return number ;
  }

  public float filter3(double inumbr) {
     //  output only to .001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 1000.) ;
       number = (float) (intermed / 1000. );
       return number ;
  }
 
  public float filter5(double inumbr) {
     //  output only to .00001
       float number ;
       int intermed ;
 
       intermed = (int) (inumbr * 100000.) ;
       number = (float) (intermed / 100000. );
       return number ;
  }
 
  public void setUnits() {   // Switching Units
       double ovs,chords,spans,aros,chos,spos ;
       double alts,ares,ards,spds ;

       alts = alt / lconv ;
       chords = chord / lconv ;
       spans = span / lconv ;
       ares = area /lconv/lconv ;
       aros = arold /lconv/lconv ;
       chos = chrdold / lconv ;
       spos = spnold / lconv ;
       ovs = vfsd / vconv ;
       ards = aread /lconv/lconv ;
       spds = sped / vconv ;

       switch (lunits) {
          case 0: {                             /* English */
            lconv = 1.;                      /*  feet    */
            vconv = .6818; vmaxa = 50.; vmaxb = 10. ;  /*  mph  */
            fconv = 1.0; fmax = 100000.; fmaxb = .5;  /* pounds   */
            pconv = 14.7  ;                   /* lb/sq in */
            break;
          }
          case 1: {                             /* Metric */
            lconv = .3048;                    /* meters */
            vconv = 1.097;  vmaxa = 80.; vmaxb = 17.;   /* km/hr  */
            fconv = 4.448 ; fmax = 500000.; fmaxb = 2.5; /* newtons */
            pconv = 101.3 ;               /* kilo-pascals */
            break ;
          }
       }
 
       alt = alts * lconv ;
       chord = chords * lconv ;
       span = spans * lconv ;
       area = ares * lconv * lconv ;
       arold = aros * lconv * lconv ;
       chrdold = chos * lconv ;
       spnold = spos * lconv ;
       vfsd  = ovs * vconv;
       aread = ards * lconv * lconv ;
       sped = spds * vconv ;

       return ;
  }

  public void loadOut() {   // output routine
     double stfact,cdragp,corfact,glide ;
     double af,bf,cf ;
     double y0,y1,y2 ;
     Double V1 ;
     double v1 ;
     float fl1 ;
//  determine output angle for dial
//     lift balance
     if (balance == 1) {
                          // stall model
        stfact = 1.0 ;
        pegged = 0 ;
        if (anflag == 1) {
            if (alfval > 10.0 ) {
               stfact = .5 + .1 * alfval - .005 * alfval * alfval ;
               if (model == 1) stfact = 1.5 ;
               if (model == 2) stfact = 1.01 - .01 * alfval  ;
               if (model == 3) {
                    if (alfval <= 16.0) stfact = 1.1 - .015 * alfval  ;
                    if (alfval > 16.0) stfact = .86 - .05 * (alfval-16.)  ;
               }
               if (model == 4) {
                    if (alfval < 15.0) stfact = 1. + .2 * alfval ;
                    if (alfval >= 15.0) stfact = 1. + .35 * alfval ;
               }
               if (model == 5) {
                    if (alfval <= 16.0) stfact = 1. + .1 * alfval ;
                    if (alfval > 16.0) stfact = 1. + .12 * alfval ;
               }
               if (model == 6) {
                    if (alfval <= 16.0) stfact = 1. + .1 * alfval ;
                    if (alfval > 16.0) stfact = 1. + .12 * alfval ;
               }
               if (model == 7) {
                    if (alfval <= 14.0) stfact = .8 ;
                    if ((alfval > 14.0) && (alfval <=17) ) stfact = .8 - (alfval - 14.0)*.07 ;
                    if (alfval > 17.0) stfact = .6 - (alfval - 17.0)*.04 ;
               }
               if (model == 8) {
                    if (alfval <= 14.0) stfact = .8 ;
                    if ((alfval > 14.0) && (alfval <=17) ) stfact = .8 - (alfval - 14.0)*.07 ;
                    if (alfval > 17.0) stfact = .6 - (alfval - 17.0)*.04 ;
               }
               if (model == 9) {
                    if (alfval < 14.0) stfact = .85 ;
                    if ((alfval >= 14.0) && (alfval <=17) ) stfact = .75 - (alfval - 14.0)*.07 ;
                    if (alfval > 17.0) stfact = .55 - (alfval - 17.0)*.04 ;
               }
               if (model == 10) stfact = .95 - (alfval-10.0) * .02;
               if (model == 11) stfact = .95 - (alfval-10.0) * .02;
               if (model == 12) stfact = .95 - (alfval-10.0) * .02;
               if (model == 15) {
                    if (alfval < 15.0) stfact = 1. + .02 * alfval ;
                    if (alfval >= 15.0) stfact = 1.5 + .2 * (alfval - 15.0) ;
               }
               if (model == 16) {
                    if (alfval < 15.0) stfact = 1. + .02 * alfval ;
                    if (alfval >= 15.0) stfact = 1.5 + .2 * (alfval - 15.0) ;
               }
               if (model == 17) {
                    if (alfval < 15.0) stfact = 1. + .02 * alfval ;
                    if (alfval >= 15.0) stfact = 1.5 + .18 * (alfval - 15.0) ;
               }
               if (model == 18) {
                    if (alfval <= 15.0) stfact = 1.0 ;
                    if (alfval > 15.0) stfact = 1.0 - (alfval - 15.0)*.04 ;
               }
               if (model == 19) {
                    if (alfval < 15.0) stfact = 1.0 ;
                    if (alfval >= 15.0) stfact = .95 - (alfval - 15.0)*.045 ;
               }
               if (model == 20) {
                    if (alfval < 14.0) stfact = 1.0 ;
                    if (alfval >= 14.0) stfact = 1.0 + (alfval - 14.0)*.1 ;
               }
               if (model == 21) {
                    if (alfval < 15.0) stfact = .95 - (alfval - 10.0)*.02  ;
                    if (alfval >= 15.0) stfact = .8 - (alfval - 15.0)*.05 ;
               }
               if (model == 23) {
                    if (alfval < 15.0) stfact = .95 - (alfval-10.0) * .02;
                    if (alfval >= 15.0) stfact = .8 - (alfval-15.0) * .05;
               }
               if (model == 24) {
                    if (alfval < 15.0) stfact = .90 - (alfval-10.0) * .03;
                    if (alfval >= 15.0) stfact = .77 - (alfval-15.0) * .05;
               }
               if (model == 25) stfact = .95 - (alfval - 10.0)*.02  ;
               if (model == 27) {
                    if (alfval < 15.0) stfact = .95 - (alfval-10.0) * .03;
                    if (alfval >= 15.0) stfact = .77 - (alfval-15.0) * .04;
               }
               if (model == 30) stfact = 1.0 ;
               if (model == 31) {
                     if (alfval < 15.0) stfact =  1.0 - (alfval - 10.0)*.04 ;
                     if (alfval >= 15.0) stfact =  .8 - (alfval - 15.0)*.03 ;
               }
               if (model == 33) stfact = .95 - (alfval - 10.0)*.025  ;
               if (model == 35) stfact = .9 - (alfval - 10.0)*.04;
               if (model == 40) {
                    if (alfval <= 13.0) stfact = 1.0 - (alfval - 10.0)*.02 ;
               }
               if (model == 41) {
                    if (alfval <= 13.0) stfact = 1.0 - (alfval - 10.0)*.02 ;
               }
               if (model == 43) {
                    if (alfval <= 16.0) stfact = 1.0 - (alfval -10.0)*.02  ;
                    if (alfval > 16.0) stfact = .88 - (alfval - 16.0)*.03 ;
               }
               if (model == 51) stfact = .9 - (alfval - 10.0)*.04;
            }
            clift = clift * stfact ;
        }
   //     correction for aspect ratio
   
        clift = clift /(1.0 + clift/(3.14159*aspr)) ;
        cdragp = 1.28 ;
 
        corfact = .5 ;
        if(model == 1) corfact = .3 ;
        if(model == 2) corfact = .5 ;
        if(model == 3) corfact = .53 ;
        if(model == 4) { 
              if (alfval <= 10.0) corfact = .3 + alfval * .006 ;
              if (alfval  > 10.0) corfact = .25 + (alfval - 10.0) * .004 ;
        }
        if(model == 5){
              if (alfval <= 10.0) corfact = .3 + alfval * .006 ;
              if (alfval  > 10.0) corfact = .25 + (alfval - 10.0) * .008 ;
        } 
        if(model == 6) {
              if (alfval <= 10.0) corfact = .3 + alfval * .006 ;
              if (alfval  > 10.0) corfact = .25 + (alfval - 10.0) * .008 ;
        }
        if(model == 7) corfact = .20 + .03*alfval ;
        if(model == 8) corfact = .22 + .03*alfval ;
        if(model == 9) corfact = .25 + .03*alfval ;
        if(model == 10) {
               if (alfval <= 6.0)  corfact = .22 + .025*alfval ;
               if (alfval > 6.0)  corfact = .37  ;
        }
        if(model == 11) {
               if (alfval <= 6.0)  corfact = .26 + .025*alfval ;
               if (alfval > 6.0)  corfact = .41  ;
        }
        if(model == 12) {
               if (alfval <= 6.0)  corfact = .28 + .025*alfval ;
               if (alfval > 6.0 && alfval < 15.0)  corfact = .43 + .005*alfval  ;
               if (alfval > 15.0)  corfact = .5 - (alfval - 15.)*.01  ;
        }
        if(model == 15) {
               if (alfval <= 10.0)  corfact = .15 + .02*alfval ;
               if (alfval > 10.0)  corfact = .35  ;
        }
        if(model == 16) {
               if (alfval <= 10.0)  corfact = .15 + .02*alfval ;
               if (alfval > 10.0)  corfact = .35  ;
        }
        if(model == 17) {
               if (alfval <= 10.0)  corfact = .15 + .02*alfval ;
               if (alfval > 10.0)  corfact = .35  ;
        }
        if(model == 18) { 
              if (alfval <= 10.0) corfact = .3 + alfval * .015 ;
              if (alfval  > 10.0) corfact = .45 + (alfval - 10.0) * .005 ;
        }
        if(model == 19) { 
              if (alfval <= 10.0) corfact = .3 + alfval * .015 ;
              if (alfval  > 10.0) corfact = .45 + (alfval - 10.0) * .005 ;
        }
        if(model == 20) { 
              if (alfval < 5.0) corfact = .2 + alfval * .015 ;
              if (alfval >= 5.0) corfact = .25 - (alfval - 5.0) * .007 ;
        }
        if(model == 21) { 
              if (alfval <= 8.0) corfact = .2 + alfval * .04 ;
              if (alfval  > 8.0) corfact = .5 + (alfval - 8.0) * .005 ;
        }
        if(model == 23) {
               if (alfval <= 10.0)  corfact = .10 + .008*alfval ;
               if (alfval > 10.0)   corfact = .2 + .005*(alfval-10.0)  ;
        }
        if(model == 24) {
               if (alfval <= 10.0)  corfact = .15 + .02*alfval ;
               if (alfval > 10.0)   corfact = .4 + .01*(alfval-10.0)  ;
        }
        if(model == 25) { 
              if (alfval < 5.0) corfact = .2 + alfval * .04 ;
              if (alfval >= 5.0) corfact = .35 - (alfval - 5.0) * .01 ;
        }
        if(model == 27) {
               if (alfval < 10.0)  corfact = .3 + .025*alfval ;
               if (alfval >= 10.0)   corfact = .52 + .02*(alfval-10.0)  ;
        }
        if(model == 30) corfact = .07 + alfval * .004 ; 
        if(model == 31) {
              if (alfval <= 6.0) corfact = .35 + alfval * .04 ; 
              if (alfval > 6.0) corfact = .60 + (alfval - 6.0) * .003 ; 
        }
        if(model == 33) { 
              if (alfval < 5.0) corfact = .4 + alfval * .04 ;
              if (alfval >= 5.0) corfact = .6 - (alfval - 5.0) * .01 ;
        }
        if(model == 35) {
               if (alfval <= 5.0)  corfact = .2 + .04*alfval ;
               if (alfval > 5.0)  corfact = .4 + .01*(alfval - 5.0)  ;
        }
        if(model == 40) { 
              if (alfval <= 10.0) corfact = .55 + alfval * .015 ;
              if (alfval  > 10.0) corfact = .7 + (alfval - 10.0) * .005 ;
              if (alfval > 13.0) pegged = 1 ;
        }
        if(model == 41) { 
              if (alfval <= 5.0) corfact = .55 + alfval * .04 ;
              if (alfval > 5.0) corfact = .55 + alfval * .025 ;
              if (alfval > 10.0) pegged = 1 ;
        }
        if(model == 43) { 
              if (alfval <= 10.0) corfact = .48 + alfval * .01 ;
              if (alfval  > 10.0) corfact = .6 + (alfval - 10.0) * .005 ;
        }
        if(model == 51) {
               if (alfval <= 6.0)  corfact = .2 + .02*alfval ;
               if (alfval > 6.0)  corfact = .3 + .01*(alfval - 6.0)  ;
        }

        clift = corfact * clift ;
 
        if (stepper == 3) {
           angout = .9 * Math.asin(clift/cdragp)/convdr ;
           if (pegged == 1) angout = 90.0 ;
        }
        if (stepper == 4) {
           angout =  Math.asin(clift/cdragp)/convdr ;
           if (pegged == 1) angout = 90.0 ;
        }
     }
//    drag balance -  curve fits
     if (balance == 2) {
        y0 = 19.5 ;
        y1 = 2.75 ;
        y2 = -2.0 ;
        if(model == 4) { 
           y0 = 19.5 ;  y1 = 2.75;  y2 = -2.0 ;
        }
        if(model == 5) { 
           y0 = 16.0 ;  y1 = 1.0;  y2 = -2.5 ;
        }
        if(model == 6) { 
           y0 = 13.5 ;  y1 = 0.75;  y2 = -2.0 ;
        }
        if(model == 7) { 
           y0 = 18.0 ;  y1 = -1.6;  y2 = -5.0 ;
        }
        if(model == 8) { 
           y0 = 12.75 ;  y1 = -3.0;  y2 = -3.75 ;
        }
        if(model == 9) { 
           y0 = 11.0 ;  y1 = -3.75;  y2 = -3.0 ;
        }
        if(model == 10) { 
           y0 = 17.5 ;  y1 = 0.0;  y2 = -2.25 ;
        }
        if(model == 11) { 
           y0 = 15.5 ;  y1 = -1.825;  y2 = -2.75 ;
        }
        if(model == 12) { 
           y0 = 14.75 ;  y1 = -3.25;  y2 = -2.25 ;
        }
        if(model == 15) { 
           y0 = 19.75 ;  y1 = 3.0;  y2 = -3.125 ;
        }
        if(model == 16) { 
           y0 = 19.75 ;  y1 = 2.0;  y2 = -4.0 ;
        }
        if(model == 17) { 
           y0 = 20.0 ;  y1 = 1.125;  y2 = -4.5 ;
        }
        if(model == 18) { 
           y0 = 12.0 ;  y1 = -2.75;  y2 = -2.125 ;
        }
        if(model == 19) { 
           y0 = 15.0 ;  y1 = -2.825;  y2 = -3.25 ;
        }
        if(model == 20) { 
           y0 = 13.0 ;  y1 = 9.0;  y2 = 6.78 ;
        }
        if(model == 21) { 
           y0 = 20.5 ;  y1 = -3.25;  y2 = -1.25 ;
        }
        if(model == 23) { 
           y0 = 20. ;  y1 = -0.125;  y2 = -1.0 ;
        }
        if(model == 24) { 
           y0 = 20. ;  y1 = 0.5;  y2 = -1.75 ;
        }
        if(model == 25) { 
           y0 = 15.5 ;  y1 = 2.75;  y2 = 4.25 ;
        }
        if(model == 27) { 
           y0 = 23.0 ;  y1 = 1.25;  y2 = -1.5 ;
        }
        if(model == 30) { 
           y0 = 38.0 ;  y1 = 18.0;  y2 = 9.5 ;
        }
        if(model == 31) { 
           y0 = 11.0 ;  y1 = -2.3;  y2 = -4.5 ;
        }
        if(model == 33) { 
           y0 = 21.0 ;  y1 = 2.75;  y2 = 3.0 ;
        }
        if(model == 35) { 
           y0 = 20.5 ;  y1 = -1.0;  y2 = 1.125 ;
        }
        if(model == 40) { 
           y0 = 15.0 ;  y1 = -1.0;  y2 = -4.25 ;
        }
        if(model == 41) { 
           y0 = 15.0 ;  y1 = -1.125;  y2 = -3.5 ;
        }
        if(model == 43) { 
           y0 = 16.75 ;  y1 = -0.5;  y2 = -5.0 ;
        }
        if(model == 51) { 
           y0 = 14.5 ;  y1 = 3.0;  y2 = -0.125 ;
        }

        cf = y0;
        bf = (4.0*y1 - y2 - 3.0*y0)/20.0 ;
        af = (y2 - 2.0*y1 + y0)/200.0 ;
        
//        v1 = bf ;
//        fl1 = (float) v1 ;
//        lq.ob.inleft.o1.setText(String.valueOf(fl1)) ;
//        v1 = af ;
//        fl1 = (float) v1 ;
//        lq.ob.inleft.o2.setText(String.valueOf(fl1)) ;

        if (stepper >= 3) {
           angout = af * alfval * alfval + bf * alfval + cf ;
              //  special for flat plate
           stfact = 1.0 ;
           if (model < 4) {
              if (anflag == 1) {
                  if (alfval > 10.0 ) {
                     if (model == 1) stfact = 1.5 ;
                     if (model == 2) stfact = 1.01 - .01 * alfval  ;
                     if (model == 3) {
                          if (alfval <= 16.0) stfact = 1.1 - .015 * alfval  ;
                          if (alfval > 16.0) stfact = .86 - .05 * (alfval-16.)  ;
                     }
                  }
                  clift = clift * stfact ;
              }
         //     correction for aspect ratio
   
              clift = clift /(1.0 + clift/(3.14159*aspr)) ;
 
              corfact = .5 ;
              if(model == 1) corfact = .3 ;
              if(model == 2) corfact = .5 ;
              if(model == 3) corfact = .53 ;
              clift = corfact * clift ;
//              if (clift < .001) clift = .001;
              angout = 90.0 - Math.atan(1.0 / clift)/convdr - alfval ;
           }
        }
     glide = angout + alfval ;
     cdrag = Math.tan(convdr * glide) ;
     }
 
// correction for area on lift coeffcient
     clift = clift * 6.0 /area ;

     return ;
  }
 
  public void loadProbe() {   // probe output routine

     pbval = 0.0 ;
     if (pboflag == 1) pbval = vel * vfsd ;           // velocity
     if (pboflag == 2) pbval = ((ps0 + pres * q0)/2116.) * pconv ; // pressure

     return ;
  }

  class Solver {
 
     Solver () {
     }

     public void setDefaults() {

        trans2 = 0 ;
        trans1 = 0 ;
        pegged = 0 ;
        balance = 1;
        group = 1 ;
        model = 1 ;
        starter = 0 ;
        stepper = 0 ;
        ysl = 80 ;
        varflg = 0 ;
        planet = 0 ;
        lunits = 0 ;
        lftout = 0 ;
        nlnc = 21 ;
        nln2 = nlnc/2 + 1 ;
        nptc = 65 ;
        npt2 = nptc/2 + 1 ;
        deltb = .5 ;
        foil = 3 ;
        flflag = 1;
        thkval = .040 ;
        thkinpt = 1.0 ;                   /* MODS 10 SEP 99 */
        camval = 0.0 ;
        caminpt = 0.0 ;
        alfval = 0.0 ;
        gamval = 0.0 ;
        spin = 0.0 ;
        spindr = 1.0 ;
        rval = 1.0 ;
        ycval = 0.0 ;
        xcval = 0.0 ;
        conflag = 1 ;                             /* MODS  2 Apr 99 */
        displ = 0 ;                              /* MODS  22 Apr 99 */
        dispp = 2 ;
        lift = 53. ;
        weight = 50. ;
        aread = 6. ;
        sped  = 15. ;
        plane = 0 ;
 
        xpval = 2.1;
        ypval = -.5 ;
        pboflag = 0 ;
        xflow = -17.0;                             /* MODS  20 Jul 99 */

        pconv = 14.7;
        pmin = .5 ;
        pmax = 1.0 ;
        fconv = 1.0 ;
        fmax = 100000. ;
        fmaxb = .50 ;
        vconv = .6818 ;
        vfsd = 0.0 ;
        vmaxa = 50. ;
        vmaxb = 10. ;
        lconv = 1.0 ;

        alt = 0.0 ;
        altmax = 50000. ;
        chrdold = chord = 5.0 ;
        spnold = span = 17.0 ;
        aspr = 6.0 ;
        arold = area = 6.0 ;
        armax = 510.01 ;
        armin = .01 ;                 /* MODS 9 SEP 99 */
 
                   //  plot defaults
        xt = 130;  yt = 80; fact = 15.0 ;
        xtp = 95; ytp = 230; factp = 25.0 ;
 
        nabs = 1; nord = 1;
        begx = -5.0;  endx = 20.0 ; ntikx = 6 ;
        begy = 0.0;   endy = 1.25  ; ntiky = 6 ;
        laby = String.valueOf("Lift Coef.");
        labx = String.valueOf("Angle of Attack");
        ntr = 0 ;
        npt = 0 ;
        nplts = 0 ;
        lines = 0 ;
 
        probflag = 2 ;
        anflag = 1 ;
        vmn = 0.0;     vmx = 50.0 ;
        almn = 0.0;    almx = 50000.0 ;
        angmn = -5.0; angmx = 20.0 ;
        camn = -25.0;  camx = 25.0 ;
        thkmn = 1.0; thkmx = 26.0 ;
        chrdmn = .1 ;  chrdmx = 10.1 ;
        spnmn = .1 ;  spnmx = 100.1 ;
        armn = .01 ;  armx = 510.01 ;

        angout = 0.0 ;
        return ;
     }

     public void getFreeStream() {    //  free stream conditions
       double hite,pvap,rgas,gama ;       /* MODS  19 Jan 00  whole routine*/

       rgas = 1718. ;                /* ft2/sec2 R */
       gama = 1.4 ;
       hite = alt/lconv ;
       if (planet == 0) {    // Earth  standard day
         if (conflag == 1) {
           if (hite <= 36152.) {           // Troposphere
              ts0 = 518.6 - 3.56 * hite/1000. ;
              ps0 = 2116. * Math.pow(ts0/518.6,5.256) ;
           }
           if (hite >= 36152. && hite <= 82345.) {   // Stratosphere
              ts0 = 389.98 ;
              ps0 = 2116. * .2236 *
                 Math.exp((36000.-hite)/(53.35*389.98)) ;
           }
           if (hite >= 82345.) {
              ts0 = 389.98 + 1.645 * (hite-82345)/1000. ;
              ps0 = 2116. *.02456 * Math.pow(ts0/389.98,-11.388) ;
           }
           rlhum = 0.0 ;
           temf = ts0 - 459.6 ;
           if (temf <= 0.0) temf = 0.0 ;                    
           presm = ps0 * 29.92 / 2116. ;
         }
         if (conflag == 2) {
            ts0 = temf + 459.6 ;
            if (temf < 0.0) {
                  temf = 0.0 ;
                  rlhum = 0.0 ;
            }
             ps0 = presm * 2116. / 29.92 ;
         }
         pvap = rlhum*(2.685+.00353*Math.pow(temf,2.245));/* Eq 1:6A  Domasch */
         rho = (ps0 - .379*pvap)/(rgas * ts0) ;  /* effect of humidty */
         rho = ps0/(53.3 * 32.17 * ts0) ;
       }

       if (planet == 1) {   // Mars - curve fit of orbiter data
         rgas = 1149. ;                /* ft2/sec2 R */
         gama = 1.29 ;

         if (hite <= 22960.) {
            ts0 = 434.02 - .548 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         if (hite > 22960.) {
            ts0 = 449.36 - 1.217 * hite/1000. ;
            ps0 = 14.62 * Math.pow(2.71828,-.00003 * hite) ;
         }
         rho = ps0/(rgas*ts0) ;
       }

       q0  = .5 * rho * vfsd * vfsd / (vconv * vconv) ;
       pt0 = ps0 + q0 ;

       return ;
     }

     public void getCirc() {   // circulation from Kutta condition
       double thet,rdm,thtm ;
       double beta,rball;
       int index;

       xcval = 0.0 ;
       switch (foil)  {
          case 0: {         /* get circulation from spin for baseball */
              rball = .1 ;         /* baseball radius = .1 ft = 1.2 in */
              gamval = 4.0 * 3.1415926 * 3.1415926 *spin * rball * rball
                                 / (vfsd/vconv) ;
              gamval = gamval * spindr ;
              ycval = .0001 ;
              break ;
          }
          case 1:  {                  /* Juokowski geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 +Math.sqrt(thkval*thkval/16.0+ycval*ycval +1.0);
              xcval = 1.0 - Math.sqrt(rval*rval - ycval*ycval) ;
              beta = Math.asin(ycval/rval)/convdr ;     /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
          case 2:  {                  /* Elliptical geometry*/
              ycval = camval / 2.0 ;
              rval = thkval/4.0 + Math.sqrt(thkval*thkval/16.0+ycval*ycval+1.0);
              beta = Math.asin(ycval/rval)/convdr ;    /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
          case 3:  {                  /* Plate geometry*/
              ycval = camval / 2.0 ;
              rval = Math.sqrt(ycval*ycval+1.0);
              beta = Math.asin(ycval/rval)/convdr ;    /* Kutta condition */
              gamval = 2.0*rval*Math.sin((alfval+beta)*convdr) ;
              break ;
          }
       }
                                                   /* geometry */
       for (index =1; index <= nptc; ++index) {
           thet = (index -1)*360./(nptc-1) ;
           xg[0][index] = rval * Math.cos(convdr * thet) + xcval ;
           yg[0][index] = rval * Math.sin(convdr * thet) + ycval ;
           rg[0][index] = Math.sqrt(xg[0][index]*xg[0][index] +
                                    yg[0][index]*yg[0][index])  ;
           thg[0][index] = Math.atan2(yg[0][index],xg[0][index])/convdr;
           xm[0][index] = (rg[0][index] + 1.0/rg[0][index])*
                        Math.cos(convdr*thg[0][index]) ;
           ym[0][index] = (rg[0][index] - 1.0/rg[0][index])*
                        Math.sin(convdr*thg[0][index]) ;
           rdm = Math.sqrt(xm[0][index]*xm[0][index] +
                           ym[0][index]*ym[0][index])  ;
           thtm = Math.atan2(ym[0][index],xm[0][index])/convdr;
           xm[0][index] = rdm * Math.cos((thtm - alfval)*convdr);
           ym[0][index] = rdm * Math.sin((thtm - alfval)*convdr);
           getVel(rval,thet) ;
           plp[index] = ((ps0 + pres * q0)/2116.) * pconv ;
           plv[index] = vel * vfsd ;
       }

       return ;
     }

     public void genFlow() {   // generate flowfield
       double rnew,thet,psv,fxg;
       int k,index;
                              /* all lines of flow  except stagnation line*/
       for (k=1; k<=nlnc; ++k) {
         psv = -.5*(nln2-1) + .5*(k-1) ;
         fxg = xflow ;
         for (index =1; index <=nptc; ++ index) {
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
           if (anflag == 1) {           // stall model
              if (alfval > 10.0 && psv > 0.0) {
                   if (xm[k][index] > 0.0) {
                      ym[k][index] = ym[k][index -1] ;
                   }
              }
              if (alfval < -10.0 && psv < 0.0) {
                   if (xm[k][index] > 0.0) {
                      ym[k][index] = ym[k][index -1] ;
                   }
              }
           }
           solve.getVel(lrg,lthg) ;
           fxg = fxg + vxdir*deltb ;
         }
       }
                                              /*  stagnation line */
       k = nln2 ;
       psv = 0.0 ;
                                              /*  incoming flow */
       for (index =1; index <= npt2; ++ index) {
           rnew = 10.0 - (10.0 - rval)*Math.sin(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                   (rnew - rval*rval/rnew)) ;
           fxg =  - rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][index]  = lxgt ;
           yg[k][index]  = lygt ;
           rg[k][index]  = lrgt ;
           thg[k][index] = lthgt ;
           xm[k][index]  = lxmt ;
           ym[k][index]  = lymt ;
       }
                                              /*  downstream flow */
       for (index = 1; index <= npt2; ++ index) {
           rnew = 10.0 + .01 - (10.0 - rval)*Math.cos(pid2*(index-1)/(npt2-1)) ;
           thet = Math.asin(.999*(psv - gamval*Math.log(rnew/rval))/
                                      (rnew - rval*rval/rnew)) ;
           fxg =   rnew * Math.cos(thet) ;
           solve.getPoints (fxg,psv) ;
           xg[k][npt2+index]  = lxgt ;
           yg[k][npt2+index]  = lygt ;
           rg[k][npt2+index]  = lrgt ;
           thg[k][npt2+index] = lthgt ;
           xm[k][npt2+index]  = lxmt ;
           ym[k][npt2+index]  = lymt ;
       }
                                              /*  stagnation point */
       xg[k][npt2]  = xcval ;
       yg[k][npt2]  = ycval ;
       rg[k][npt2]  = Math.sqrt(xcval*xcval+ycval*ycval) ;
       thg[k][npt2] = Math.atan2(ycval,xcval)/convdr ;
       xm[k][npt2]  = (xm[k][npt2+1] + xm[k][npt2-1])/2.0 ;
       ym[k][npt2]  = (ym[0][nptc/4+1] + ym[0][nptc/4*3+1])/2.0 ;
                                /*  compute lift coefficient */
       leg = xcval - Math.sqrt(rval*rval - ycval*ycval) ;
       teg = xcval + Math.sqrt(rval*rval - ycval*ycval) ;
       lem = leg + 1.0/leg ;
       tem = teg + 1.0/teg ;
       chrd = tem - lem ;
       clift = gamval*4.0*3.1415926/chrd ;

       return ;
     }

     public void getPoints(double fxg, double psv) {   // flow in x-psi
       double radm,thetm ;                /* MODS  20 Jul 99  whole routine*/
       double fnew,ynew,yold,rfac,deriv ;
       double xold,xnew,thet ;
       double rmin,rmax ;
       int iter,isign;
                       /* get variables in the generating plane */
                           /* iterate to find value of yg */
       ynew = 10.0 ;
       yold = 10.0 ;
       if (psv < 0.0) ynew = -10.0 ;
       if (Math.abs(psv) < .001 && alfval < 0.0) ynew = rval ;
       if (Math.abs(psv) < .001 && alfval >= 0.0) ynew = -rval ;
       fnew = 0.1 ;
       iter = 1 ;
       while (Math.abs(fnew) >= .00001 && iter < 25) {
           ++iter ;
           rfac = fxg*fxg + ynew*ynew ;
           if (rfac < rval*rval) rfac = rval*rval + .01 ;
           fnew = psv - ynew*(1.0 - rval*rval/rfac)
                  - gamval*Math.log(Math.sqrt(rfac)/rval) ;
           deriv = - (1.0 - rval*rval/rfac)
               - 2.0 * ynew*ynew*rval*rval/(rfac*rfac)
               - gamval * ynew / rfac ;
           yold = ynew ;
           ynew = yold  - .5*fnew/deriv ;
       }
       lyg = yold ;
                                     /* rotate for angle of attack */
       lrg = Math.sqrt(fxg*fxg + lyg*lyg) ;
       lthg = Math.atan2(lyg,fxg)/convdr ;
       lxgt = lrg * Math.cos(convdr*(lthg + alfval)) ;
       lygt = lrg * Math.sin(convdr*(lthg + alfval)) ;
                              /* translate cylinder to generate airfoil */
       lxgt = lxgt + xcval ;
       lygt = lygt + ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt) ;
       lthgt = Math.atan2(lygt,lxgt)/convdr ;
                               /*  Kutta-Joukowski mapping */
       lxm = (lrgt + 1.0/lrgt)*Math.cos(convdr*lthgt) ;
       lym = (lrgt - 1.0/lrgt)*Math.sin(convdr*lthgt) ;
                              /* tranforms for view fixed with free stream */
                /* take out rotation for angle of attack mapped and cylinder */
       radm = Math.sqrt(lxm*lxm+lym*lym) ;
       thetm = Math.atan2(lym,lxm)/convdr ;
       lxmt = radm*Math.cos(convdr*(thetm-alfval)) ;
       lymt = radm*Math.sin(convdr*(thetm-alfval)) ;

       lxgt = lxgt - xcval ;
       lygt = lygt - ycval ;
       lrgt = Math.sqrt(lxgt*lxgt + lygt*lygt)  ;
       lthgt = Math.atan2(lygt,lxgt)/convdr;
       lxgt = lrgt * Math.cos((lthgt - alfval)*convdr);
       lygt = lrgt * Math.sin((lthgt - alfval)*convdr);

       return ;
     }
 
     public void getVel(double radius, double theta) {  //velocity and pressure 
      double ur,uth,jake1,jake2,jakesq ;
      double xloc,yloc,thrad,alfrad ;

      thrad = convdr * theta ;
      alfrad = convdr * alfval ;
                                /* get x, y location in cylinder plane */
      xloc = radius * Math.cos(thrad) ;
      yloc = radius * Math.sin(thrad) ;
                                /* velocity in cylinder plane */
      ur  = Math.cos(thrad-alfrad)*(1.0-(rval*rval)/(radius*radius)) ;
      uth = -Math.sin(thrad-alfrad)*(1.0+(rval*rval)/(radius*radius))
                            - gamval/radius;
      usq = ur*ur + uth*uth ;
      vxdir = ur * Math.cos(thrad) - uth * Math.sin(thrad) ; // MODS  20 Jul 99 
                                /* translate to generate airfoil  */
      xloc = xloc + xcval ;
      yloc = yloc + ycval ;
                                   /* compute new radius-theta  */
      radius = Math.sqrt(xloc*xloc + yloc*yloc) ;
      thrad  = Math.atan2(yloc,xloc) ;
                                   /* compute Joukowski Jacobian  */
      jake1 = 1.0 - Math.cos(2.0*thrad)/(radius*radius) ;
      jake2 = Math.sin(2.0*thrad)/(radius*radius) ;
      jakesq = jake1*jake1 + jake2*jake2 ;
      if (Math.abs(jakesq) <= .01) jakesq = .01 ;  /* protection */
      vsq = usq / jakesq ;
          /* vel is velocity ratio - pres is coefficient  (p-p0)/q0   */
      if (foil > 0) {
           vel = Math.sqrt(vsq) ;
           pres = 1.0 - vsq ;
      }
      if (foil == 0) {
           vel = Math.sqrt(usq) ;
           pres = 1.0 - usq ;
      }
      return ;
    }

    public void getProbe () { /* all of the information needed for the probe */
      double prxg;
      int index;
                       /* get variables in the generating plane */
      if (Math.abs(ypval) < .01) ypval = .05 ;
      solve.getPoints (xpval,ypval) ;

      solve.getVel(lrg,lthg) ;
      loadProbe() ;

      pxg = lxgt ;
      pyg = lygt ;
      prg = lrgt ;
      pthg = lthgt ;
      pxm = lxmt ;
      pym = lymt ;
                                    /* smoke */
      if (pboflag == 3 ) {
        prxg = xpval ;
        for (index =1; index <=nptc; ++ index) {
          solve.getPoints (prxg,ypval) ;
          xg[19][index] = lxgt ;
          yg[19][index] = lygt ;
          rg[19][index] = lrgt ;
          thg[19][index] = lthgt ;
          xm[19][index] = lxmt ;
          ym[19][index] = lymt ;
          if (anflag == 1) {           // stall model
             if (xpval > 0.0) {
                if (alfval > 10.0 && ypval > 0.0) {
                   ym[19][index] = ym[19][1] ;
                }
                if (alfval < -10.0 && ypval < 0.0) {
                     ym[19][index] = ym[19][1] ;
                }
             }
             if (xpval < 0.0) {
                if (alfval > 10.0 && ypval > 0.0) {
                   if (xm[19][index] > 0.0) {
                       ym[19][index] = ym[19][index-1] ;
                   }
                }
                if (alfval < -10.0 && ypval < 0.0) {
                   if (xm[19][index] > 0.0) {
                     ym[19][index] = ym[19][index-1] ;
                   }
                }
             }
          }
          solve.getVel(lrg,lthg) ;
          prxg = prxg + vxdir*deltb ;
        }
      }
      return ;
    }
  }
 
  class Lq extends Panel {
     Foil outerparent ;
     Viewer view ;
     Ob ob ;

     Lq (Foil target) { 
         outerparent = target ;
         setLayout(new GridLayout(2,1,5,5)) ;

         view  = new Viewer(outerparent) ;
         ob = new Ob(outerparent) ;

         add(view) ;
         add(ob) ;
     }

     class Ob extends Panel {
        Foil outerparent ;
        Inleft inleft ;
        Inright inright ;

        Ob (Foil target) {

           outerparent = target ;
           setLayout(new GridLayout(1,2,5,5)) ;

           inleft = new Inleft(outerparent) ;
           inright = new Inright(outerparent) ;

           add(inleft) ;
           add(inright) ;
        }

        class Inleft extends Panel {
           Foil outerparent ;
           Ipa ipa ;
           TextField o1,o2 ;
           Button bt1,bt2,bt3,bt4,bt5,bt6 ;

           Inleft (Foil target) {
            outerparent = target ;
            setLayout(new GridLayout(7,1,2,2)) ;

            bt1 = new Button("Step 1: Select Model       ") ;
            bt1.setBackground(Color.blue) ;
            bt1.setForeground(Color.white) ;

            bt2 = new Button("Step 2: Set Angle of Attack") ;
            bt2.setBackground(Color.red) ;
            bt2.setForeground(Color.white) ;

            bt3 = new Button("Step 3: Start Tunnel       ") ;
            bt3.setBackground(Color.red) ;
            bt3.setForeground(Color.white) ;

            bt4 = new Button("Step 4: Adjust for Drag    ") ;
            bt4.setBackground(Color.red) ;
            bt4.setForeground(Color.white) ;
 
            bt5 = new Button("Step 5: Record Data") ;
            bt5.setBackground(Color.red) ;
            bt5.setForeground(Color.white) ;
 
            bt6 = new Button("        Reset              ") ;
            bt6.setBackground(Color.yellow) ;
            bt6.setForeground(Color.black) ;
 
            ipa = new Ipa(outerparent) ;

            add(ipa) ;
            add(bt1) ;
            add(bt2) ;
            add(bt3) ;
            add(bt4) ;
            add(bt5) ;
            add(bt6) ;
          }

          public boolean action(Event evt, Object arg) {
            int i,j ;

            if(evt.target instanceof Button) {
               String label = (String)arg ;
               if(label.equals("Step 1: Select Model       ")) {
                   starter = 1 ;
                   stepper = 1 ;
                   if (nplts > 0) {
                      ncurv[ntr] = npt ;
                      legnd[ntr] = model ;
                      sq.reorder() ;
                   }
                   angout = 0.0 ;
                   vfsd = 0.0 ;
                   alfval = 0.0  ;
                   ++nplts ;
                   ntr = nplts ;
                   npt = 0 ;
                   lines = 0 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.white) ;
                   bt1.setForeground(Color.blue) ;
                   bt2.setBackground(Color.red) ;
                   bt2.setForeground(Color.white) ;
                   bt3.setBackground(Color.red) ;
                   bt3.setForeground(Color.white) ;
                   bt4.setBackground(Color.red) ;
                   bt4.setForeground(Color.white) ;
                   bt5.setBackground(Color.red) ;
                   bt5.setForeground(Color.white) ;
                   if (balance == 1) {
                      partimg = getImage(getCodeBase(),"wrightbal.gif");
                      ob.inright.lab1.setText("Lift Balance");
                   }
                   if (balance == 2) {
                      partimg = getImage(getCodeBase(),"dbalance.gif");
                      ob.inright.lab1.setText("Drag Balance");
                   }
                   inright.btp.l1.setText("Photo from Wright State University");
               }
               if(label.equals("Step 2: Set Angle of Attack")) {
                   starter = 1 ;
                   stepper = 2 ;
                   angout = 0.0 ;
                   vfsd = 0.0 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.blue) ;
                   bt1.setForeground(Color.white) ;
                   bt2.setBackground(Color.white) ;
                   bt2.setForeground(Color.red) ;
                   bt3.setBackground(Color.red) ;
                   bt3.setForeground(Color.white) ;
                   bt4.setBackground(Color.red) ;
                   bt4.setForeground(Color.white) ;
                   bt5.setBackground(Color.red) ;
                   bt5.setForeground(Color.white) ;
               }
               if(label.equals("Step 3: Start Tunnel       ")) {
                   starter = 1 ;
                   stepper = 3 ;
                   trans1 = 0 ;
                   vfsd = 20.0 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.blue) ;
                   bt1.setForeground(Color.white) ;
                   bt2.setBackground(Color.red) ;
                   bt2.setForeground(Color.white) ;
                   bt3.setBackground(Color.white) ;
                   bt3.setForeground(Color.red) ;
                   bt4.setBackground(Color.red) ;
                   bt4.setForeground(Color.white) ;
                   bt5.setBackground(Color.red) ;
                   bt5.setForeground(Color.white) ;
                   inright.btp.l1.setText(" ");
               }
               if(label.equals("Step 4: Adjust for Drag    ")) {
                   starter = 1 ;
                   stepper = 4 ;
                   trans2 = 0 ;
                   vfsd = 20.0 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.blue) ;
                   bt1.setForeground(Color.white) ;
                   bt2.setBackground(Color.red) ;
                   bt2.setForeground(Color.white) ;
                   bt3.setBackground(Color.red) ;
                   bt3.setForeground(Color.white) ;
                   bt4.setBackground(Color.white) ;
                   bt4.setForeground(Color.red) ;
                   bt5.setBackground(Color.red) ;
                   bt5.setForeground(Color.white) ;
               }
               if(label.equals("Step 5: Record Data")) {
                   starter = 1 ;
                   stepper = 5 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.blue) ;
                   bt1.setForeground(Color.white) ;
                   bt2.setBackground(Color.red) ;
                   bt2.setForeground(Color.white) ;
                   bt3.setBackground(Color.red) ;
                   bt3.setForeground(Color.white) ;
                   bt4.setBackground(Color.red) ;
                   bt4.setForeground(Color.white) ;
                   bt5.setBackground(Color.white) ;
                   bt5.setForeground(Color.red) ;
               }
               if(label.equals("        Reset              ")) {
                   stepper = 0 ;
                   starter = 0 ;
                   balance = 1 ;
                   ipa.bt0l.setBackground(Color.black) ;
                   ipa.bt0l.setForeground(Color.white) ;
                   ipa.bt0d.setBackground(Color.black) ;
                   ipa.bt0d.setForeground(Color.white) ;
                   bt1.setBackground(Color.blue) ;
                   bt1.setForeground(Color.white) ;
                   bt2.setBackground(Color.red) ;
                   bt2.setForeground(Color.white) ;
                   bt3.setBackground(Color.red) ;
                   bt3.setForeground(Color.white) ;
                   bt4.setBackground(Color.red) ;
                   bt4.setForeground(Color.white) ;
                   bt5.setBackground(Color.red) ;
                   bt5.setForeground(Color.white) ;
                   partimg = getImage(getCodeBase(),"tunnel.gif");
                   ob.inright.lab1.setText("Wright 1901 Wind Tunnel");
                   inright.btp.l1.setText("Photo from Wright State University");
                   angout = 0.0 ;
                   vfsd = 0.0 ;
                   alfval = 0.0  ;
                   ntr = 0 ;
                   npt = 0 ;
                   nplts = 0 ;
                   for (i = 0; i <= 4; ++i) {
                       ncurv[i] = 0 ;
                       legnd[i] = 0 ;
                       xlegn[i] = 100.0 ;
                       ylegn[i] = 0.0 ;
                       for (j = 0; j <= 49; ++j) {
                          pltx[i][j] = 0.0 ;
                          plty[i][j] = 0.0 ;
                       }  
                   }
               }

               computeFlow() ;
               if(label.equals("Step 5: Record Data")) loadPlot() ;
               lq.ob.inright.l.repaint() ;
               return true ;
            }
            else return false ;
          } // Handler
 
          void loadPlot() {
            if (npt == 25 ) return ;
            ++npt ;
            switch (nord) {
               case 1: plty[ntr][npt] = clift ; break ;
               case 2: plty[ntr][npt] = cdrag ; break ;
            }
            pltx[ntr][npt] = alfval ;
          }

          class Ipa extends Panel {
             Foil outerparent ;
             Button bt0l,bt0d ;
      
             Ipa (Foil target) {
      
               outerparent = target ;
               setLayout(new GridLayout(1,2,2,2)) ;

               bt0l = new Button("Lift Balance") ;
               bt0l.setBackground(Color.black) ;
               bt0l.setForeground(Color.white) ;
 
               bt0d = new Button("Drag Balance") ;
               bt0d.setBackground(Color.black) ;
               bt0d.setForeground(Color.white) ;
 
               add(bt0l) ;
               add(bt0d) ;
             }

             public boolean action(Event evt, Object arg) {
                int i, j ;

               if(evt.target instanceof Button) {
                  String label = (String)arg ;
                  if(label.equals("Lift Balance")) {
                      starter = 1 ;
                      stepper = 0 ;
                      balance = 1 ;
                      nord = 1 ;
                      laby = String.valueOf("Lift Coef.");
                      begy = 0.0;   endy = 1.25  ; ntiky = 6 ;
                      bt0l.setBackground(Color.white) ;
                      bt0l.setForeground(Color.black) ;
                      bt0d.setBackground(Color.black) ;
                      bt0d.setForeground(Color.white) ;
                      inleft.bt1.setBackground(Color.blue) ;
                      inleft.bt1.setForeground(Color.white) ;
                      inleft.bt2.setBackground(Color.red) ;
                      inleft.bt2.setForeground(Color.white) ;
                      inleft.bt3.setBackground(Color.red) ;
                      inleft.bt3.setForeground(Color.white) ;
                      inleft.bt4.setBackground(Color.red) ;
                      inleft.bt4.setForeground(Color.white) ;
                      inleft.bt5.setBackground(Color.red) ;
                      inleft.bt5.setForeground(Color.white) ;
                      partimg = getImage(getCodeBase(),"wrightbal.gif");
                      inright.lab1.setText("Lift Balance");
                      inright.btp.l1.setText("Photo from Wright State University");
                  }
                  if(label.equals("Drag Balance")) {
                      starter = 1 ;
                      stepper = 0 ;
                      balance = 2 ;
                      nord = 2 ;
                      laby = String.valueOf("Drag/Lift Ratio");
                      begy = 0.0;   endy = .5  ; ntiky = 6 ;
                      bt0l.setBackground(Color.black) ;
                      bt0l.setForeground(Color.white) ;
                      bt0d.setBackground(Color.white) ;
                      bt0d.setForeground(Color.black) ;
                      inleft.bt1.setBackground(Color.blue) ;
                      inleft.bt1.setForeground(Color.white) ;
                      inleft.bt2.setBackground(Color.red) ;
                      inleft.bt2.setForeground(Color.white) ;
                      inleft.bt3.setBackground(Color.red) ;
                      inleft.bt3.setForeground(Color.white) ;
                      inleft.bt4.setBackground(Color.red) ;
                      inleft.bt4.setForeground(Color.white) ;
                      inleft.bt5.setBackground(Color.red) ;
                      inleft.bt5.setForeground(Color.white) ;
                      partimg = getImage(getCodeBase(),"dbalance.gif");
                      inright.lab1.setText("Drag Balance");
                      inright.btp.l1.setText("Photo from Wright State University");
                  }
                  angout = 0.0 ;
                  vfsd = 0.0 ;
                  alfval = 0.0  ;
                  ntr = 0 ;
                  npt = 0 ;
                  nplts = 0 ;
                  for (i = 0; i <= 4; ++i) {
                      ncurv[i] = 0 ;
                      legnd[i] = 0 ;
                      xlegn[i] = 100.0 ;
                      ylegn[i] = 0.0 ;
                      for (j = 0; j <= 49; ++j) {
                         pltx[i][j] = 0.0 ;
                         plty[i][j] = 0.0 ;
                      }  
                  }
                  lq.ob.inright.l.repaint() ;
                  computeFlow() ;
                  return true ;
               }
               else return false ;
             } // Text Handler
          }  // Ipa
        }  // Inleft

        class Inright extends Panel {
            Foil outerparent ;
            L l;
            Label lab1 ;
            Btp btp ;

            Inright (Foil target) {

             outerparent = target ;
             setLayout(new BorderLayout(1,1)) ;

             l = new L(outerparent) ;
             btp = new Btp(outerparent) ;
             lab1 = new Label("Wright 1901 Wind Tunnel", Label.CENTER) ;

             add("South",btp) ;
             add("Center",l) ;
             add("North",lab1) ;
           }

           class Btp extends Panel {
              Foil outerparent ;
              Label l1,l2 ;

              Btp (Foil target) {
               outerparent = target ;
               setLayout(new GridLayout(1,1,1,1)) ;

               l1 = new Label("Photo from Wright State University", Label.CENTER) ;

               add(l1) ;
             }
           } // end Btp

           class L extends Canvas  {
              Foil outerparent ;

              L (Foil target) {
                setBackground(Color.black) ;
              }

              public void update(Graphics g) {
                ob.inright.l.paint(g) ;
              }

              public void paint(Graphics g) {
                int ex,ey,index ;
                double tick ;
                Color brown ;
   
                brown = new Color((float) (.81),(float) (.59),(float) (.40)) ;

                if (stepper >= 3) {
                   off3Gg.setColor(brown) ;
                   off3Gg.fillRect(0,0,300,300) ;

                   off3Gg.setColor(Color.gray) ;
                   off3Gg.fillArc(-75,20,200,200,0,90) ;

         // tick marks
                   off3Gg.setColor(Color.yellow) ;
                   for (index = 0; index <= 8; ++index) {
                      tick = 5. + index*10. ;
                      ex = 25 + (int) (100.0 * Math.sin(convdr * tick)) ;
                      ey = 120 - (int) (100.0 * Math.cos(convdr* tick)) ;
                      off3Gg.drawLine(25,120,ex,ey) ;
                   }

                   off3Gg.setColor(Color.gray) ;
                   off3Gg.fillArc(-65,30,180,180,0,90) ;

                   off3Gg.setColor(Color.yellow) ;
                   for (index = 0; index <= 9; ++index) {
                      tick = index*10. ;
                      ex = 25 + (int) (100.0 * Math.sin(convdr * tick)) ;
                      ey = 120 - (int) (100.0 * Math.cos(convdr* tick)) ;
                      off3Gg.drawLine(25,120,ex,ey) ;
                   }

                   off3Gg.setColor(Color.gray) ;
                   off3Gg.fillArc(-60,35,170,170,0,90) ;
   
                   off3Gg.setColor(Color.yellow) ;
                   if (balance == 1) off3Gg.drawString("0",25,20) ;
                   if (balance == 2) off3Gg.drawString("25",25,20) ;
                   off3Gg.setColor(Color.yellow) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *10.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *10.0)) ;
                   if(balance == 1) off3Gg.drawString("10",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("20",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *20.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *20.0)) ;
                   if(balance == 1) off3Gg.drawString("20",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("15",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *30.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *30.0)) ;
                   if(balance == 1) off3Gg.drawString("30",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("10",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *40.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *40.0)) ;
                   if(balance == 1) off3Gg.drawString("40",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("5",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *50.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *50.0)) ;
                   if(balance == 1) off3Gg.drawString("50",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("0",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *60.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *60.0)) ;
                   if(balance == 1) off3Gg.drawString("60",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("-5",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *70.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *70.0)) ;
                   if(balance == 1) off3Gg.drawString("70",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("-10",ex,ey) ;
                   ex = 25 + (int) (100.0 * Math.sin(convdr *80.0)) ;
                   ey = 120 - (int) (100.0 * Math.cos(convdr *80.0)) ;
                   if(balance == 1) off3Gg.drawString("80",ex,ey) ;
                   if(balance == 2) off3Gg.drawString("-15",ex,ey) ;
                   if(balance == 1) off3Gg.drawString("90",125,120) ;
                   if(balance == 2) off3Gg.drawString("-20",125,120) ;
   
                   off3Gg.setColor(Color.black) ;
                   if (balance == 1) {
                      ex = 25 + (int) (110.0 * Math.sin(convdr *
                              angdraw)) ;
                      ey = 120 - (int) (110.0 * Math.cos(convdr *
                           angdraw)) ;
                   }
                   if (balance == 2) {
                      ex = 25 + (int) (110.0 * Math.sin(convdr *
                              (50.0 - 2.0*angdraw))) ;
                      ey = 120 - (int) (110.0 * Math.cos(convdr *
                              (50.0 - 2.0*angdraw))) ;
                   }
                   off3Gg.drawLine(25,120,ex,ey) ;
 
                   off3Gg.fillOval(22,117,6,6);

                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("Dial Angle =",20,140) ;
                   off3Gg.drawString(String.valueOf(filter3(angdraw)),90,140) ;
                   if (balance == 1 && ((angout - angdraw) <= .01)) {
                     off3Gg.drawString("Lift Coef. =",20,155) ;
                     off3Gg.drawString(String.valueOf(filter3(clift)),90,155) ;
                   }
                   if (balance == 2 && ((angout - angdraw) <= .01)) {
                     off3Gg.drawString("Drag/Lift Ratio =",20,155) ;
                     off3Gg.drawString(String.valueOf(filter3(cdrag)),120,155) ;
                   }
                   off3Gg.setColor(Color.yellow) ;
                   off3Gg.drawString("Model #",120,25) ;
                   off3Gg.drawString(String.valueOf(model),165,25) ;
                }
 
                if (stepper <= 2) {
                   off3Gg.setColor(Color.white) ;
                   off3Gg.fillRect(0,0,300,300) ;

                   off3Gg.drawImage(partimg,0,0,this) ;
                }

                g.drawImage(offImg3,0,0,this) ;
             }
           } //Lor2a
        }  // Inright
     }  // Outprb

     class Viewer extends Canvas  
         implements Runnable{
        Foil outerparent ;
        Thread runner ;
        Point locate,anchor;
   
        Viewer (Foil target) {
            setBackground(Color.black) ;
            runner = null ;
        }    

        public Insets insets() {
           return new Insets(0,10,0,10) ;
        }
 

        public boolean mouseDrag(Event evt, int x, int y) {
           handle(x,y) ;
           return true;
        }
 
        public void handle(int x, int y) {
         // determine location
            if (stepper == 2) {
               if (x < 105 ) {   
                  if (y >= 75  && y <= 100) {
                     locate = new Point(x,y) ;
                     ysl = locate.y ;
                     alfval = angmn + (double) ((ysl - 75)*(angmx - angmn)/25.) ; 
                     computeFlow();
                  }
               }
            }
        }

        public boolean mouseUp(Event evt, int x, int y) {
           handleb(x,y) ;
           return true;
        }

        public void handleb(int x, int y) {
          if (stepper == 1) {
             if (y >= 10 && y < 20) {  // tunnel image
                if (x >= 20 && x < 100) {  
                   partimg = getImage(getCodeBase(),"tunnel.gif");
                }
             }
             lq.ob.inright.l.repaint() ;
             computeFlow() ;
          }
        }

        public void start() {
           if (runner == null) {
              runner = new Thread(this) ;
              runner.start() ;
           }
           antim = 0 ;                              /* MODS  21 JUL 99 */
           ancol = 1 ;                              /* MODS  27 JUL 99 */
        }
   
        public void run() {
          int timer ;
    
          timer = 100 ;
          while (true) {
             ++ trans2 ;
             ++ trans1 ;
             ++ antim ;
             try { Thread.sleep(timer); }
             catch (InterruptedException e) {}
             lq.view.repaint() ;
             if (antim == 3) {
                antim = 0;
                ancol = - ancol ;               /* MODS 27 JUL 99 */
             }
             timer = 135 - (int) (.227 *vfsd/vconv) ;
             angdraw = trans1 * angout / 30.0 ;
             if (trans1 > 30) angdraw = angout ;
             lq.ob.inright.l.repaint() ;
             angdraw2 = angout - trans2 * angout / 30.0 ;
             if (trans2 > 30) angdraw2 = 0.0 ;
          }
        }
   
        public void update(Graphics g) {
           lq.view.paint(g) ;
        }
    
        public void paint(Graphics g) {
           int i,j,k,n ;
           int xlabel,ylabel,ind,inmax,inmin ;
           int xcen,ycen,lstrut,wide,tall ;
           int xtm,ytm ;
           int xh1,yh1,xh2,yh2 ;
           int xh3,yh3,xh4,yh4 ;
           int xh5,yh5,xh6,yh6 ;
           int deg1,deg2 ;
           int exes[] = new int[8] ;
           int whys[] = new int[8] ;
           double offx,scalex,offy,scaley,waste,incy,incx;
           double xtrans,ytrans;
           int xl,yl ;
           int camx[] = new int[19] ;
           int camy[] = new int[19] ;
           Color col,brown ;
   
           col = new Color(0,0,0) ;
           if(planet == 0) col = Color.cyan ;
           brown = new Color((float) (.81),(float) (.59),(float) (.40)) ;
           off1Gg.setColor(brown) ;
           off1Gg.fillRect(0,0,500,400) ;
//    y adjustment
           yl = 10 ;

           xtm = xt ;
           ytm = yt ;
// lift balance
           if (balance == 1) {
// mounting bracket
              xcen = 170 ;
              ycen = yl+165 ;
              lstrut = 40 ;
   
              wide = 16 ;
              tall = 160 ;
              exes[0] = xcen - wide/2 ;
              whys[0] = yl + ycen + wide/2 ;
              exes[1] = exes[0];
              whys[1] = whys[0] - tall - wide/2 ;
              exes[2] = exes[1] + wide ;
              whys[2] = whys[1] ;
              exes[3] = exes[2];
              whys[3] = whys[0] ;
              off1Gg.setColor(Color.darkGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
// dial
              off1Gg.setColor(Color.gray) ;
              off1Gg.fillArc(120,yl + 120,100,100,0,90) ;
              off1Gg.setColor(Color.black) ;
              off1Gg.drawLine(xcen,ycen,
                   (int) (xcen + 40 * Math.sin(convdr*angdraw)),
                   (int) (ycen - 40 * Math.cos(convdr*angdraw))) ;
// lower arms 
              wide = 12 ;
              tall = 4 ;
              exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw))) ;
              whys[0] = (int) (ycen  -116 + tall*Math.cos(convdr*(45.0 - angdraw))) ;
              exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                          - tall*Math.sin(convdr*(45.0 + angdraw))) ;
              whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) -116
                          + tall*Math.sin(convdr*(45.0 + angdraw))) ;
              exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                          - tall*Math.sin(convdr*(45.0 - angdraw))) ;
              whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) -116
                          - tall*Math.sin(convdr*(45.0 - angdraw))) ;
              exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw))) ;
              whys[3] = (int) (ycen -116 - tall*Math.cos(convdr*(45.0 + angdraw))) ; ;
              off1Gg.setColor(Color.gray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
   
              wide = 12 ;
              tall = 4 ;
              exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw))) ;
              whys[0] = (int) (ycen + tall*Math.cos(convdr*(45.0 - angdraw))) ;
              exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                             - tall*Math.sin(convdr*(45.0 + angdraw))) ;
              whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw)
                          + tall*Math.sin(convdr*(45.0 + angdraw))) ;
              exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                          - tall*Math.sin(convdr*(45.0 - angdraw))) ;
              whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw)
                          - tall*Math.sin(convdr*(45.0 - angdraw))) ;
              exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw))) ;
              whys[3] = (int) (ycen - tall*Math.cos(convdr*(45.0 + angdraw))) ; ;
              off1Gg.setColor(Color.gray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
// lower cross beam 
              wide = 8 ;
              tall = 120 ;
              exes[0] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) -wide/2) ;
              whys[0] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) +wide/2) ;
              exes[1] = exes[0];
              whys[1] = whys[0] - tall -wide/2;
              exes[2] = exes[1] + wide ;
              whys[2] = whys[1] ;
              exes[3] = exes[2];
              whys[3] = whys[0] ;
              off1Gg.setColor(Color.gray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
              off1Gg.setColor(Color.black) ;
              off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
              off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
              off1Gg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
              off1Gg.drawLine(exes[3],whys[3],exes[0],whys[0]) ;
  // plates 
              exes[0] = exes[3];
              whys[0] = whys[2] + 5;
              exes[1] = exes[0] + 5;
              whys[1] = whys[0] + 10;
              exes[2] = exes[0];
              whys[2] = whys[1] + 10;
              off1Gg.fillPolygon(exes,whys,3) ;
   
              exes[0] = exes[3];
              whys[0] = whys[2] + 10;
              exes[1] = exes[0] + 3;
              whys[1] = whys[0] + 10;
              exes[2] = exes[0];
              whys[2] = whys[1] + 10;
              off1Gg.fillPolygon(exes,whys,3) ;
   
              exes[0] = exes[3];
              whys[0] = whys[2] + 10;
              exes[1] = exes[0] + 3;
              whys[1] = whys[0] + 10;
              exes[2] = exes[0];
              whys[2] = whys[1] + 10;
              off1Gg.fillPolygon(exes,whys,3) ;
   
              exes[0] = exes[3];
              whys[0] = whys[2] + 10;
              exes[1] = exes[0] + 5;
              whys[1] = whys[0] + 10;
              exes[2] = exes[0];
              whys[2] = whys[1] + 10;
              off1Gg.fillPolygon(exes,whys,3) ;
   
//   upper cross beam
              if (stepper >= 4 && pegged == 0) {
                 xtm = (int) (xt + lstrut*(1.0 - Math.cos(convdr*angdraw2))) ;
                 ytm = yl+(int) (yt - lstrut*Math.sin(convdr*angdraw2)) ;
                 wide = 12 ;
                 tall = 6 ;
                 exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 whys[0] = (int) (ycen  -116 + tall*Math.cos(convdr*(45.0 - angdraw2))) ;
                 exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw2) 
                                - tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw2) -116
                                + tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw2) 
                          - tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw2) -116
                                - tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 whys[3] = (int) (ycen -116 - tall*Math.cos(convdr*(45.0 + angdraw2))) ; ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
         
                 wide = 12 ;
                 tall = 6 ;
                 exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 whys[0] = (int) (ycen + tall*Math.cos(convdr*(45.0 - angdraw2))) ;
                 exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw2) 
                                - tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw2)
                                + tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw2) 
                                - tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw2)
                                - tall*Math.sin(convdr*(45.0 - angdraw2))) ;
                 exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw2))) ;
                 whys[3] = (int) (ycen - tall*Math.cos(convdr*(45.0 + angdraw2))) ; ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
      
                 wide = 10 ;
                 tall = 125 ;
                 exes[0] = (int) (xcen - lstrut*Math.cos(convdr*angdraw2) -wide/2) ;
                 whys[0] = (int) (ycen - lstrut*Math.sin(convdr*angdraw2) +wide/2) ;
                 exes[1] = exes[0];
                 whys[1] = whys[0] - tall -wide/2;
                 exes[2] = exes[1] + wide ;
                 whys[2] = whys[1] ;
                 exes[3] = exes[2];
                 whys[3] = whys[0] ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                 off1Gg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
                 off1Gg.drawLine(exes[3],whys[3],exes[0],whys[0]) ;
   
                 off1Gg.setColor(Color.black) ;
                 off1Gg.fillOval(xcen-2,ycen-2,4,4);
                 off1Gg.fillOval(xcen-2,ycen-118,4,4);
                 off1Gg.fillOval((int) (xcen-lstrut*Math.cos(convdr*angdraw2)-2),
                          (int) (ycen-lstrut*Math.sin(convdr*angdraw2)-2),4,4);
                 off1Gg.fillOval((int) (xcen-lstrut*Math.cos(convdr*angdraw2)-2),
                             (int) (ycen-lstrut*Math.sin(convdr*angdraw2)-118),4,4);
              }
              else {
                 xtm = (int) (xt + lstrut*(1.0 - Math.cos(convdr*angdraw))) ;
                 ytm = yl+(int) (yt - lstrut*Math.sin(convdr*angdraw)) ;
   
                 wide = 12 ;
                 tall = 6 ;
                 exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 whys[0] = (int) (ycen  -116 + tall*Math.cos(convdr*(45.0 - angdraw))) ;
                 exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                                - tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) -116
                                + tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                          - tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) -116
                                - tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 whys[3] = (int) (ycen -116 - tall*Math.cos(convdr*(45.0 + angdraw))) ; ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
         
                 wide = 12 ;
                 tall = 6 ;
                 exes[0] = (int) (xcen + tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 whys[0] = (int) (ycen + tall*Math.cos(convdr*(45.0 - angdraw))) ;
                 exes[1] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                                - tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 whys[1] = (int) (ycen - lstrut*Math.sin(convdr*angdraw)
                                + tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 exes[2] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) 
                                - tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 whys[2] = (int) (ycen - lstrut*Math.sin(convdr*angdraw)
                                - tall*Math.sin(convdr*(45.0 - angdraw))) ;
                 exes[3] = (int) (xcen + tall*Math.sin(convdr*(45.0 + angdraw))) ;
                 whys[3] = (int) (ycen - tall*Math.cos(convdr*(45.0 + angdraw))) ; ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
      
                 wide = 10 ;
                 tall = 125 ;
                 exes[0] = (int) (xcen - lstrut*Math.cos(convdr*angdraw) -wide/2) ;
                 whys[0] = (int) (ycen - lstrut*Math.sin(convdr*angdraw) +wide/2) ;
                 exes[1] = exes[0];
                 whys[1] = whys[0] - tall -wide/2;
                 exes[2] = exes[1] + wide ;
                 whys[2] = whys[1] ;
                 exes[3] = exes[2];
                 whys[3] = whys[0] ;
                 off1Gg.setColor(Color.lightGray) ;
                 off1Gg.fillPolygon(exes,whys,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                 off1Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
                 off1Gg.drawLine(exes[2],whys[2],exes[3],whys[3]) ;
                 off1Gg.drawLine(exes[3],whys[3],exes[0],whys[0]) ;
   
                 off1Gg.setColor(Color.black) ;
                 off1Gg.fillOval(xcen-2,ycen-2,4,4);
                 off1Gg.fillOval(xcen-2,ycen-118,4,4);
                 off1Gg.fillOval((int) (xcen-lstrut*Math.cos(convdr*angdraw)-2),
                          (int) (ycen-lstrut*Math.sin(convdr*angdraw)-2),4,4);
                 off1Gg.fillOval((int) (xcen-lstrut*Math.cos(convdr*angdraw)-2),
                             (int) (ycen-lstrut*Math.sin(convdr*angdraw)-118),4,4);
              }
           }
// drag balance
           if (balance == 2) {
              xcen = 130 ;
              ycen = yl + 120 ;
              
              glidang = alfval + angdraw ;
              xtm = (int) (xcen - 25 * Math.sin(convdr*glidang)) ;
              ytm = (int) (ycen - 25 * Math.cos(convdr*glidang)) ;

              lstrut = 80 ;
              wide = 16 ;

              xh1 = (int) (xcen + lstrut * Math.cos(convdr*alfval)) ;
              yh1 = (int) (ycen - lstrut * Math.sin(convdr*alfval)) ;
              xh2 = (int) (xcen - lstrut * Math.cos(convdr*alfval)) ;
              yh2 = (int) (ycen + lstrut * Math.sin(convdr*alfval)) ;
// mounting bracket
              exes[0] = (int) (xh1 + wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[0] = (int) (yh1 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              exes[1] = (int) (xh1 + wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[1] = (int) (yh1 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[2] = (int) (xh2 - wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[2] = (int) (yh2 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[3] = (int) (xh2 - wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[3] = (int) (yh2 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              off1Gg.setColor(Color.darkGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
// dial
              deg1 = (int) (135 + alfval) ;
              off1Gg.setColor(Color.gray) ;
              off1Gg.fillArc(xh1-40,yh1-40,80,80,deg1,90) ;
              off1Gg.setColor(Color.black) ;
              off1Gg.drawLine(xh1,yh1,
                 (int) (xh1 - 40 * Math.cos(convdr*glidang)),
                 (int) (yh1 + 40 * Math.sin(convdr*glidang))) ;
              tall = 45 ;
              xh3 = (int) (xh1 - tall * Math.sin(convdr*glidang)) ;
              yh3 = (int) (yh1 - tall * Math.cos(convdr*glidang)) ;
              xh4 = (int) (xh1 + tall * Math.sin(convdr*glidang)) ;
              yh4 = (int) (yh1 + tall * Math.cos(convdr*glidang)) ;
              xh5 = (int) (xh2 - tall * Math.sin(convdr*glidang)) ;
              yh5 = (int) (yh2 - tall * Math.cos(convdr*glidang)) ;
              xh6 = (int) (xh2 + tall * Math.sin(convdr*glidang)) ;
              yh6 = (int) (yh2 + tall * Math.cos(convdr*glidang)) ;
// arms 
              wide = 12 ;

              exes[0] = (int) (xh3 + wide/2 * Math.cos(convdr*(45.0 +glidang)));
              whys[0] = (int) (yh3 - wide/2 * Math.sin(convdr*(45.0 +glidang)));
              exes[1] = (int) (xh4 + wide/2 * Math.cos(convdr*(45.0 -glidang)));
              whys[1] = (int) (yh4 + wide/2 * Math.sin(convdr*(45.0 -glidang)));
              exes[2] = (int) (xh4 - wide/2 * Math.cos(convdr*(45.0 +glidang)));
              whys[2] = (int) (yh4 + wide/2 * Math.sin(convdr*(45.0 +glidang)));
              exes[3] = (int) (xh3 - wide/2 * Math.cos(convdr*(45.0 -glidang)));
              whys[3] = (int) (yh3 - wide/2 * Math.sin(convdr*(45.0 -glidang)));
              off1Gg.setColor(Color.lightGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;

              exes[0] = (int) (xh5 + wide/2 * Math.cos(convdr*(45.0 +glidang)));
              whys[0] = (int) (yh5 - wide/2 * Math.sin(convdr*(45.0 +glidang)));
              exes[1] = (int) (xh6 + wide/2 * Math.cos(convdr*(45.0 -glidang)));
              whys[1] = (int) (yh6 + wide/2 * Math.sin(convdr*(45.0 -glidang)));
              exes[2] = (int) (xh6 - wide/2 * Math.cos(convdr*(45.0 +glidang)));
              whys[2] = (int) (yh6 + wide/2 * Math.sin(convdr*(45.0 +glidang)));
              exes[3] = (int) (xh5 - wide/2 * Math.cos(convdr*(45.0 -glidang)));
              whys[3] = (int) (yh5 - wide/2 * Math.sin(convdr*(45.0 -glidang)));
              off1Gg.setColor(Color.lightGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;
// beams 
              wide = 8 ;

              exes[0] = (int) (xh3 + wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[0] = (int) (yh3 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              exes[1] = (int) (xh3 + wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[1] = (int) (yh3 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[2] = (int) (xh5 - wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[2] = (int) (yh5 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[3] = (int) (xh5 - wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[3] = (int) (yh5 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              off1Gg.setColor(Color.lightGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;

              exes[0] = (int) (xh4 + wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[0] = (int) (yh4 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              exes[1] = (int) (xh4 + wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[1] = (int) (yh4 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[2] = (int) (xh6 - wide/2 * Math.cos(convdr*(45.0 + alfval)));
              whys[2] = (int) (yh6 + wide/2 * Math.sin(convdr*(45.0 - alfval)));
              exes[3] = (int) (xh6 - wide/2 * Math.cos(convdr*(45.0 - alfval)));
              whys[3] = (int) (yh6 - wide/2 * Math.sin(convdr*(45.0 + alfval)));
              off1Gg.setColor(Color.lightGray) ;
              off1Gg.fillPolygon(exes,whys,4) ;

              off1Gg.setColor(Color.black) ;
              off1Gg.fillOval(xh1-2,yh1-2,4,4);
              off1Gg.fillOval(xh2-2,yh2-2,4,4);
              off1Gg.fillOval(xh3-2,yh3-2,4,4);
              off1Gg.fillOval(xh4-2,yh4-2,4,4);
              off1Gg.fillOval(xh5-2,yh5-2,4,4);
              off1Gg.fillOval(xh6-2,yh6-2,4,4);
           }

// flowfield 
         if(stepper > 0) {
           if (displ <= 3) {  // Side View
            if (vfsd > .01) {
                                               /* plot airfoil flowfield */
             for (j=1; j<=nln2-1; ++j) {           /* lower half */
                exes[1] = (int) (fact*(-xm[j][1])) + xtm ;
                whys[1] = (int) (fact*(-ym[j][1])) + ytm ;
                for (i=2 ; i<= nptc; ++i) {
                   exes[0] = exes[1] ;
                   whys[0] = whys[1] ;
                   exes[1] = (int) (fact*(-xm[j][i])) + xtm ;
                   whys[1] = (int) (fact*(-ym[j][i])) + ytm ;
                   if (displ == 2) {                   /* MODS  21 JUL 99 */
                     off1Gg.setColor(Color.yellow) ;
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                   if (displ == 1 && (i/3*3 == i) ) {
                     off1Gg.setColor(col) ;
                     for (n=1 ; n <= 4 ; ++n) {
                        if(i == 6 + (n-1)*9) off1Gg.setColor(Color.red);
                     }
                     if(i/9*9 == i) off1Gg.setColor(Color.white);
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                   if (displ == 0 && ((i-antim)/3*3 == (i-antim)) ) {
                     if (ancol == -1) {          /* MODS  27 JUL 99 */
                       if((i-antim)/6*6 == (i-antim))off1Gg.setColor(col);
                       if((i-antim)/6*6 != (i-antim))off1Gg.setColor(Color.white);
                     }
                     if (ancol == 1) {          /* MODS  27 JUL 99 */
                       if((i-antim)/6*6 == (i-antim))off1Gg.setColor(Color.white);
                       if((i-antim)/6*6 != (i-antim))off1Gg.setColor(col);
                     }
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                }
             }
             for (j=nln2+1; j<=nlnc; ++j) {          /* upper half */
                exes[1] = (int) (fact*(-xm[j][1])) + xtm ;
                whys[1] = (int) (fact*(-ym[j][1])) + ytm ;
                for (i=2 ; i<= nptc; ++i) {
                   exes[0] = exes[1] ;
                   whys[0] = whys[1] ;
                   exes[1] = (int) (fact*(-xm[j][i])) + xtm ;
                   whys[1] = (int) (fact*(-ym[j][i])) + ytm ;
                   if (displ == 2) {                     /* MODS  21 JUL 99 */
                     off1Gg.setColor(col) ;
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                   if (displ == 1 && (i/3*3 == i) ) {
                     off1Gg.setColor(col);   /* MODS  27 JUL 99 */
                     for (n=1 ; n <= 4 ; ++n) {
                        if(i == 6 + (n-1)*9) off1Gg.setColor(Color.red);
                     }
                     if(i/9*9 == i) off1Gg.setColor(Color.white);
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                   if (displ == 0 && ((i-antim)/3*3 == (i-antim)) ) {
                     if (ancol == -1) {          /* MODS  27 JUL 99 */
                       if((i-antim)/6*6 == (i-antim))off1Gg.setColor(col);
                       if((i-antim)/6*6 != (i-antim))off1Gg.setColor(Color.white);
                     }
                     if (ancol == 1) {          /* MODS  27 JUL 99 */
                       if((i-antim)/6*6 == (i-antim))off1Gg.setColor(Color.white);
                       if((i-antim)/6*6 != (i-antim))off1Gg.setColor(col);
                     }
                     off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                   }
                }
             }
            }
         }
     // draw the airfoil geometry
             exes[1] = (int) (fact*(-xm[0][npt2])) + xtm ;
             whys[1] = (int) (fact*(-ym[0][npt2])) + ytm ;
             for (i=1 ; i<= npt2-1; ++i) {
                exes[0] = exes[1] ;
                whys[0] = whys[1] ;
                exes[1] = (int) (fact*(-xm[0][npt2-i])) + xtm ;
                whys[1] = (int) (fact*(-ym[0][npt2-i])) + ytm ;
                off1Gg.setColor(Color.black) ;
                off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
             }
             if (model == 24 || model == 27 || model == 33 || model == 40){
                exes[1] = (int) (fact*(-xm[0][npt2])) + xtm ;
                whys[1] = (int) (fact*(-ym[0][npt2])) + ytm + 30 ;
                for (i=1 ; i<= npt2-1; ++i) {
                   exes[0] = exes[1] ;
                   whys[0] = whys[1] ;
                   exes[1] = (int) (fact*(-xm[0][npt2-i])) + xtm ;
                   whys[1] = (int) (fact*(-ym[0][npt2-i])) + ytm + 30 ;
                   off1Gg.setColor(Color.black) ;
                   off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                }
             }
             if (model == 27 || model == 43) {
                exes[1] = (int) (fact*(-xm[0][npt2])) + xtm ;
                whys[1] = (int) (fact*(-ym[0][npt2])) + ytm + 15 ;
                for (i=1 ; i<= npt2-1; ++i) {
                   exes[0] = exes[1] ;
                   whys[0] = whys[1] ;
                   exes[1] = (int) (fact*(-xm[0][npt2-i])) + xtm ;
                   whys[1] = (int) (fact*(-ym[0][npt2-i])) + ytm + 15 ;
                   off1Gg.setColor(Color.black) ;
                   off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                }
             }
             if (model == 41) {
                exes[1] = (int) (fact*(-xm[0][npt2])) + xtm ;
                whys[1] = (int) (fact*(-ym[0][npt2])) + ytm + 45 ;
                for (i=1 ; i<= npt2-1; ++i) {
                   exes[0] = exes[1] ;
                   whys[0] = whys[1] ;
                   exes[1] = (int) (fact*(-xm[0][npt2-i])) + xtm ;
                   whys[1] = (int) (fact*(-ym[0][npt2-i])) + ytm + 45 ;
                   off1Gg.setColor(Color.black) ;
                   off1Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
                }
             }
          }

   //   some notes
          if (stepper <= 1) {
               off1Gg.setColor(Color.white) ;
               off1Gg.drawString("1901 Wind Tunnel",20,yl + 20) ;
               off1Gg.setColor(Color.black) ;
               off1Gg.drawString("Test Section",30,yl + 35) ;
               off1Gg.drawString("(Overhead View)",22,yl + 50) ;

               off1Gg.setColor(Color.black) ;
               off1Gg.drawString("Direction of",220,yl + 65) ;
               off1Gg.drawString("Air Flow",240,yl + 80) ;
                off1Gg.drawLine(255,yl+90,285,yl + 90) ;
                exes[0] = 255 ;  exes[1] = 245; exes[2] = 255 ;
                whys[0] = yl + 85 ;   whys[1] = yl + 90;  whys[2] = yl + 95 ;
                off1Gg.fillPolygon(exes,whys,3) ;

             if (stepper == 1) {
                off1Gg.setColor(Color.yellow) ;
                off1Gg.drawString("Model #",25,yl + 65) ;
                off1Gg.drawString(String.valueOf(model),70,yl + 65) ;
 //               off1Gg.drawString("Mounted Here",10,125) ;
                off1Gg.drawLine(85,yl + 60,100,yl + 80) ;
                exes[0] = 90 ;  exes[1] = 100; exes[2] = 102 ;
                whys[0] = yl + 75 ;   whys[1] = yl + 80;  whys[2] = yl + 70 ;
                off1Gg.fillPolygon(exes,whys,3) ;
            }
          }
          if (stepper == 2) {
               off1Gg.setColor(Color.yellow) ;
               off1Gg.drawString("Set Angle of Attack",30,yl + 20) ;
               off1Gg.drawString("Model #",30,yl + 35) ;
               off1Gg.drawString(String.valueOf(model),75,yl + 35) ;
               off1Gg.drawString("Move Cursor onto Wilbur's Arm",210,yl + 20) ;
               off1Gg.drawString("Hold Down on the Button",210,yl + 35) ;
               off1Gg.drawString("Move the Cursor Up or Down",210,yl + 50) ;
               off1Gg.drawString("Angle of Attack =",220,yl + 65) ;
               off1Gg.drawString(String.valueOf(alfval),310,yl + 65) ;
 //  Wilbur's arm
               off1Gg.setColor(Color.black) ;
               off1Gg.fillRect(0,yl + ysl-15,80,30) ;
               off1Gg.setColor(Color.white) ;
               off1Gg.fillRect(80,yl + ysl-8,25,12) ;
               off1Gg.setColor(Color.black) ;
               off1Gg.drawLine(95,yl + ysl-4,105,yl+ysl-4) ;
               off1Gg.drawLine(97,yl + ysl,105,yl+ysl) ;
               off1Gg.setColor(Color.white) ;
               off1Gg.fillRect(80,yl + ysl-12,15,4) ;
               off1Gg.fillRect(80,yl + ysl+4,22,4) ;
               off1Gg.setColor(Color.black) ;
               off1Gg.drawLine(85,yl + ysl-9,95,yl+ysl-9) ;
               off1Gg.drawLine(97,yl + ysl+4,102,yl+ysl+4) ;
          }
          if (stepper == 4) {
               if (balance ==1 && trans2 <= 30) {
 //  Wilbur's arm
                 off1Gg.setColor(Color.black) ;
                 off1Gg.fillRect(0,yl + ytm-15,xtm-10,30) ;
                 off1Gg.setColor(Color.white) ;
                 off1Gg.fillRect(xtm-10,yl + ytm-8,25,12) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(xtm+5,yl + ytm-4,xtm+15,yl+ytm-4) ;
                 off1Gg.drawLine(xtm+7,yl + ytm,xtm+15,yl+ytm) ;
                 off1Gg.setColor(Color.white) ;
                 off1Gg.fillRect(xtm-10,yl + ytm-12,15,4) ;
                 off1Gg.fillRect(xtm-10,yl + ytm+4,22,4) ;
                 off1Gg.setColor(Color.black) ;
                 off1Gg.drawLine(xtm-5,yl + ytm-9,xtm+5,yl+ytm-9) ;
                 off1Gg.drawLine(xtm+7,yl + ytm+4,xtm+12,yl+ytm+4) ;
               }
               if (trans2 > 30 && pegged == 1) {
                 off1Gg.setColor(Color.yellow) ;
                 off1Gg.drawString("No Adjustment Possible",10,yl + 160) ;
                 off1Gg.drawString("at this angle",20,yl + 175) ;
               }
          }
          g.drawImage(offImg1,0,0,this) ;   
       }
    } // end Viewer
  }  // end lokpnl
 
  class Sq extends Canvas
         implements Runnable{
     Foil outerparent ;
     Thread runner ;

     Sq (Foil target) { 
       setBackground(Color.black) ;
       runner = null ;
     }

     public boolean mouseUp(Event evt, int x, int y) {
        handleb(x,y) ;
        return true;
     }

     public void handleb(int x, int y) {

       if (stepper == 1) {
          lq.ob.inright.btp.l1.setText("Photo from Franklin Institute");
          if (group == 1) {
            if(x <= 190) {
             if (y >= 15 && y < 45) {  // model 1
                model = 1 ;
                foil = 3 ;
                camval = 0.0 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod1.gif");
             }
             if (y >= 45 && y < 75) {  // model 2
                model = 2 ;
                foil = 3 ;
                camval = 0.0 ;
                aspr = 4.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod2.gif");
             }
             if (y >= 75 && y < 105) {  // model 3
                model = 3 ;
                foil = 3 ;
                camval = 0.0 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod3.gif");
             }
             if (y >= 105 && y < 135) {  // model 4
                model = 4 ;
                foil = 3 ;
                camval = 0.33333 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod4.gif");
             }
             if (y >= 135 && y < 165) {  // model 5
                model = 5 ;
                foil = 3 ;
                camval = 0.25 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod5.gif");
             }
             if (y >= 165 && y < 195) {  // model 6
                model = 6 ;
                foil = 3 ;
                camval = 0.2 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod6.gif");
             }
             if (y >= 195 && y < 225) {  // model 7
                model = 7 ;
                foil = 3 ;
                camval = 0.333 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod7.gif");
             }
             if (y >= 225 && y < 255) {  // model 8
                model = 8 ;
                foil = 3 ;
                camval = 0.25 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod8.gif");
             }
             if (y >= 255 && y < 285) {  // model 9
                model = 9 ;
                foil = 3 ;
                camval = 0.2 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod9.gif");
             }
             if (y >= 285 && y < 315) {  // model 10
                model = 10 ;
                foil = 1 ;
                camval = 0.33333 ;
                thkval = .5 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod10.gif");
             }
             if (y >= 315 && y < 345) {  // model 11
                model = 11 ;
                foil = 1 ;
                camval = 0.25 ;
                thkval = .5 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod11.gif");
             }
             if (y >= 345 && y < 375) {  // model 12
                model = 12 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 6.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod12.gif");
             }
           }
           if (x >= 190) {
             if (y >= 15 && y < 45) {  // model 15
                model = 15 ;
                foil = 1 ;
                camval = 0.333 ;
                thkval = .5 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod15.gif");
             }
             if (y >= 45 && y < 75) {  // model 16
                model = 16 ;
                foil = 1 ;
                camval = 0.25 ;
                thkval = .5 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod16.gif");
             }
             if (y >= 75 && y < 105) {  // model 17
                model = 17 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 1.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"nonav.gif");
             }
             if (y >= 105 && y < 135) {  // model 18
                model = 18 ;
                foil = 3 ;
                camval = 0.2 ;
                aspr = 4.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod18.gif");
             }
             if (y >= 135 && y < 165) {  // model 19
                model = 19 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 4.0 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod19.gif");
             }
             if (y >= 165 && y < 195) {  // model 20
                model = 20 ;
                foil = 1 ;
                camval = 0.7 ;
                thkval = .5 ;
                aspr = 4.7 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod20.gif");
             }
             if (y >= 195 && y < 225) {  // model 21
                model = 21 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 4.7 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod21.gif");
             }
             if (y >= 225 && y < 255) {  // model 23
                model = 23 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 6.75 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod23.gif");
             }
             if (y >= 255 && y < 285) {  // model 25
                model = 25 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 8.0 ;
                angout = 0.0 ;
                area = 4.0 ;
                partimg = getImage(getCodeBase(),"mod25.gif");
             }
             if (y >= 285 && y < 315) {  // model 30
                model = 30 ;
                foil = 1 ;
                camval = 0.7 ;
                thkval = 1.0 ;
                aspr = 4.1 ;
                area = 5.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod30.gif");
             }
             if (y >= 315 && y < 345) {  // model 31
                model = 31 ;
                foil = 1 ;
                camval = 0.33 ;
                thkval = .5 ;
                aspr = 4.64 ;
                area = 8.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod31.gif");
             }
             if (y >= 345 && y < 375) {  // model 35
                model = 35 ;
                foil = 1 ;
                camval = 0.37 ;
                thkval = .5 ;
                aspr = 6.0 ;
                area = 6.00 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod35.gif");
             }
             if (y >= 375 && y < 405) {  // model 51
                model = 51 ;
                foil = 1 ;
                camval = 0.37 ;
                thkval = .5 ;
                aspr = 4.0 ;
                area = 6.25 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod51.gif");
             }
           }
          }
          if (group == -1) {
            if(x <= 190) {
             if (y >= 15 && y < 75) {  // model 24
                model = 24 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 6.7 ;
                area = 6.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod23.gif");
             }
             if (y >= 75 && y < 135) {  // model 27
                model = 27 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 6.7 ;
                area = 9.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod23.gif");
             }
             if (y >= 135 && y < 195) {  // model 33
                model = 33 ;
                foil = 1 ;
                camval = 0.2 ;
                thkval = .5 ;
                aspr = 8.0 ;
                area = 8.0 ;
                angout = 0.0 ;
                partimg = getImage(getCodeBase(),"mod25.gif");
             }
            }
            if(x >= 190) {
                if (y >= 15 && y < 75) {  // model 40
                   model = 40 ;
                   foil = 1 ;
                   camval = 0.2 ;
                   thkval = .5 ;
                   aspr = 4. ;
                   area = 12.0 ;
                   angout = 0.0 ;
                   partimg = getImage(getCodeBase(),"mod18.gif");
                }
                if (y >= 75 && y < 135) {  // model 41
                   model = 41 ;
                   foil = 1 ;
                   camval = 0.2 ;
                   thkval = .5 ;
                   aspr = 4.0 ;
                   area = 12.0 ;
                   angout = 0.0 ;
                   partimg = getImage(getCodeBase(),"mod18.gif");
                }
                if (y >= 135 && y < 195) {  // model 43
                   model = 43 ;
                   foil = 1 ;
                   camval = 0.2 ;
                   thkval = .5 ;
                   aspr = 4.0 ;
                   area = 12.0 ;
                   angout = 0.0 ;
                   partimg = getImage(getCodeBase(),"mod18.gif");
                }
           }
          }
          if (x >= 10 && x <= 180) {
             if (y >= 382 && y <= 397) {
                group = -group;
             }
          }
          sq.repaint() ;
          lq.ob.inright.l.repaint() ;
          computeFlow() ;
       }
       if (stepper == 5) {
          if (x >= 40 && x <= 115) {    // line through data
             if (y >= 338 && y <= 353) {
                 lines = 1 ;
                 ncurv[ntr] = npt;
                 legnd[ntr] = model ;
                 reorder();
                 sq.repaint() ;
             }
          }
       }
     }  // end handleb

     public void reorder() {
        int item,i;
        double tempx,tempy;

        for (item=1; item<=npt-1; ++item) {
              for (i=item+1; i<=npt; ++i) {
                if (pltx[ntr][i] < pltx[ntr][item]) {
                   tempx = pltx[ntr][item];
                   tempy = plty[ntr][item];
                   pltx[ntr][item] = pltx[ntr][i];
                   plty[ntr][item] = plty[ntr][i];
                   pltx[ntr][i] = tempx;
                   plty[ntr][i] = tempy;
                }
              }
         }
         xlegn[ntr] = pltx[ntr][npt] + .5 ;
         ylegn[ntr] = plty[ntr][npt] ;
         return;
     }

     public void start() {
        if (runner == null) {
           runner = new Thread(this) ;
           runner.start() ;
        }
     }

     public void run() {
       int timer ;

       timer = 100 ;
       while (true) {
          try { Thread.sleep(timer); }
          catch (InterruptedException e) {}
          sq.repaint() ;
       }
     }

     public void update(Graphics g) {
       sq.paint(g) ;
     }

     public void paint(Graphics g) {
       int i,j,k,n ;
       int xlabel,ylabel,ind,inmax,inmin ;

       int ex,ey,index ;
       int exes[] = new int[8] ;
       int whys[] = new int[8] ;
       double offx,scalex,offy,scaley,waste,incy,incx;
       double xl,yl;
 
       off2Gg.setColor(Color.blue) ;
       off2Gg.fillRect(0,0,380,500) ;

    if(stepper >= 2) {   // record data draw a plot
       if (ntikx < 2) ntikx = 2 ;     /* protection 13June96 */
       if (ntiky < 2) ntiky = 2 ;
       offx = 0.0 - begx ;
       scalex = 10.0/(endx-begx) ;
       incx = (endx-begx)/(ntikx-1);
       offy = 0.0 - begy ;
       scaley = 8.5/(endy-begy) ;
       incy = (endy-begy)/(ntiky-1) ;
                                                 /* draw axes */
       off2Gg.setColor(Color.white) ;
       exes[0] = (int) (factp* 0.0) + xtp ;
       whys[0] = (int) (factp* -8.5) + ytp ;
       exes[1] = (int) (factp* 0.0) + xtp ;
       whys[1] = (int) (factp* 0.0) + ytp ;
       exes[2] = (int) (factp* 10.0) + xtp ;
       whys[2] = (int) (factp* 0.0) + ytp ;
       off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       off2Gg.drawLine(exes[1],whys[1],exes[2],whys[2]) ;
  
       xlabel = (int) (-90.0) + xtp ;   /*  label y axis */
       ylabel = (int) (factp*-4.0) + ytp ;
       off2Gg.drawString(laby,xlabel,ylabel) ;
                                                       /* add tick values */
       for (ind= 1; ind<= ntiky; ++ind){
             xlabel = (int) (-50.0) + xtp ;
             yl = begy + (ind-1) * incy ;
             ylabel = (int) (factp* -scaley*(yl + offy)) + ytp ;
             off2Gg.drawString(String.valueOf(filter3(yl)),xlabel,ylabel);
       }
       xlabel = (int) (factp*3.0) + xtp ;    /* label x axis */
       ylabel = (int) (40.0) + ytp ;
       off2Gg.drawString(labx,xlabel,ylabel-10) ;
                                                      /* add tick values */
       for (ind= 1; ind<= ntikx; ++ind){
             ylabel = (int) (15.) + ytp ;
             xl = begx + (ind-1) * incx ;
             xlabel = (int) (factp*(scalex*(xl + offx) -.05)) + xtp ;
             off2Gg.drawString(String.valueOf(xl),xlabel,ylabel) ;
       }

       if (ntr == 0) off2Gg.drawString("You must select an airfoil model - Step 1",150,175) ; 
       if (ntr == 6) off2Gg.drawString("Too many plots - Push Reset button",150,175) ; 
       if (npt >= 25) off2Gg.drawString("Too many points - Push Reset button",150,175) ; 

       if(lines == 0) {
          for (i=1; i<=npt; ++i) {                      /* plot new data */
              xlabel = (int) (factp*scalex*(offx+pltx[ntr][i])) + xtp ;
              ylabel = (int) (factp* -scaley*(offy+plty[ntr][i])) + ytp ;
              off2Gg.drawString("*",xlabel,ylabel+5) ;
          }
       }
 
       for (j=1; j<=nplts; ++j) {
         if(j==ntr && lines == 0) {
            waste =1. ;
         }
         else {
            exes[1] = (int) (factp*scalex*(offx+pltx[j][1])) + xtp ;
            whys[1] = (int) (factp* -scaley*(offy+plty[j][1])) + ytp ;
            for (i=1; i<=ncurv[j]; ++i) {
                 exes[0] = exes[1] ;
                 whys[0] = whys[1] ;
                 exes[1] = (int) (factp*scalex*(offx+pltx[j][i])) + xtp ;
                 whys[1] = (int) (factp* -scaley*(offy+plty[j][i])) + ytp ;
                 off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
            }
            xlabel = (int) (factp*scalex*(offx+xlegn[j])) + xtp ;
            ylabel = (int) (factp* -scaley*(offy+ylegn[j])) + ytp ;
            off2Gg.drawString(String.valueOf(legnd[j]),xlabel,ylabel) ;
         }
       }

       off2Gg.drawString("Click on 'Step 3' to start the tunnel and watch the dial.",20,275) ;
       off2Gg.drawString("Click on 'Step 4' to adjust the lift balance.",20,290) ;
       off2Gg.drawString("Click on 'Step 5' to record your data on the graph.(25 max)",20,305) ;
       off2Gg.drawString("Repeat Steps 2 through 5 to produce a plot.",20,320) ;
       off2Gg.drawString("When your plot is complete:",20,335) ;
       off2Gg.setColor(Color.white) ;
       off2Gg.fillRect(40,338,75,15) ;
       off2Gg.setColor(Color.black) ;
       off2Gg.drawString("CLICK HERE",40,350) ;
       off2Gg.setColor(Color.white) ;
       off2Gg.drawString("to draw a line through your data.",120,350) ;
       off2Gg.drawString("Select another model beginning with Step 1.",20,370) ;
       off2Gg.drawString("Make another plot by repeating the sequence.",20,385) ;
       off2Gg.drawString("Using your plots you can compare the performance of five models.",20,405) ;
       off2Gg.drawString("Like the Wright brothers, you can then select the best design.",20,420) ;
    }
    if(stepper == 0) {
       off2Gg.setColor(Color.white) ;
       off2Gg.drawString("Wright 1901  Wind Tunnel  Simulator",80,40) ;
       if (starter == 0) {
          off2Gg.drawString("With this simulator you can duplicate the 1901 wind",20,70) ;
          off2Gg.drawString("tunnel tests of the Wright Brothers.",20,85) ;
          off2Gg.drawString("The brothers built the wind tunnel shown in the photo to",20,105) ;
          off2Gg.drawString("measure the LIFT and DRAG of wing models which they",20,120) ;
          off2Gg.drawString("made in their bicycle shop at Dayton, Ohio.",20,135) ;
          off2Gg.drawString("The tunnel had a fan on the front and a long open box on the back.",20,150) ;
          off2Gg.drawString("The box section was about six feet long and one and a half",20,165) ;
          off2Gg.drawString("feet on a side.",20,180) ;

          off2Gg.drawString("The brothers built two BALANCES to measure lift and drag.",20,200) ;
          off2Gg.drawString("Each wing model was tested on both the LIFT BALANCE",20,215) ;
          off2Gg.drawString("and the DRAG BALANCE.",20,230) ;
          off2Gg.drawString("You begin the simulation by chosing which variable you wish",20,250) ;
          off2Gg.drawString("to measure.",20,265) ;
          off2Gg.drawString("LIFT?  or  DRAG?",20,285) ;
          off2Gg.drawString("Make your selection using the black buttons at the far left.",20,305) ;
          off2Gg.drawString("Click on either Lift Balance  or Drag Balance",20,330) ;
       }
       if (starter == 1) {
          if (balance == 1) {
             off2Gg.drawString("You have selected the LIFT BALANCE shown in the photo.",20,70) ;
             off2Gg.drawString("In the photo the balance is viewed from upstream in the tunnel",20,85) ;
             off2Gg.drawString("The wing model is mounted on the top of the balance and",20,100) ;
             off2Gg.drawString("the reference drag plates are shown at the bottom.",20,115) ;
             off2Gg.drawString("The output dial is at the lower left of the picture.",20,130) ;
             off2Gg.drawString("The computer drawing at the left shows a view looking down",20,145) ;
             off2Gg.drawString("onto the balance.",20,160) ;
             off2Gg.drawString("In the experiments you will be measuring the LIFT COEFFICIENT.",20,180) ;
             off2Gg.drawString("The lift coefficient is the ratio of the lift of the model",20,195) ;
             off2Gg.drawString("to the drag on the reference plates. The lift coefficient",20,210) ;
             off2Gg.drawString("depends on the shape of the model and the angle of attack.",20,225) ;
             off2Gg.drawString("The angle of attack is the angle formed by the direction of",20,240) ;
             off2Gg.drawString("the wind and the length of the model.",20,255) ;
             off2Gg.drawString("The larger the lift coefficient, the higher the lift generated",20,270) ;
             off2Gg.drawString("by the airfoil. For the lift coefficient:",20,285) ;
             off2Gg.drawString("LARGE VALUES ARE BETTER THAN SMALL VALUES.",20,300) ;
          }
          if (balance == 2) {
             off2Gg.drawString("You have selected the DRAG BALANCE shown in the photo.",20,70) ;
             off2Gg.drawString("In the photo the balance is viewed from the side of the tunnel.",20,85) ;
             off2Gg.drawString("The wing model is mounted on the top of the balance.",20,100) ;
             off2Gg.drawString("The output dial is shown in the middle right of the photo.",20,115) ;
             off2Gg.drawString("The computer drawing at the left shows a view looking down",20,130) ;
             off2Gg.drawString("onto the balance.",20,145) ;
             off2Gg.drawString("In the experiments you will be measuring the ratio of the",20,165) ;
             off2Gg.drawString("DRAG to the LIFT of the wing models. The value of the ratio",20,180) ;
             off2Gg.drawString("depends on the shape of the model and the angle of attack.",20,195) ;
             off2Gg.drawString("The angle of attack is the angle formed by the direction of",20,210) ;
             off2Gg.drawString("the wind and the length of the model. The drag to lift ratio",20,225) ;
             off2Gg.drawString("is directly related to how well an aircraft glides.",20,240) ;
             off2Gg.drawString("The smaller the drag to lift ratio, the smaller the drag generated",20,255) ;
             off2Gg.drawString("by the wing and the farther the aircraft will fly across",20,270) ;
             off2Gg.drawString("the ground as it loses height. For the drag to lift ratio:",20,285) ;
             off2Gg.drawString("SMALL VALUES ARE BETTER THAN LARGE VALUES.",20,300) ;
          }
          off2Gg.drawString("You must now select a wing MODEL for testing.",20,335) ;
          off2Gg.drawString("Click on the blue button labeled Step 1 at the far left",20,350) ;
          off2Gg.drawString("and your choices will appear in this window. Click on your ",20,365) ;
          off2Gg.drawString("choice and the model will be installed on the balance.",20,380) ;
       }
    }
// show the models
    if(stepper == 1) {
       off2Gg.setColor(Color.white) ;
       off2Gg.drawString("#",4,12) ;
       off2Gg.drawString("Shape",20,12) ;
       off2Gg.drawString("Area-sq in",70,12) ;
       off2Gg.drawString("Cam",137,12) ;
       off2Gg.drawString("AR",170,12) ;

       off2Gg.drawString("#",194,12) ;
       off2Gg.drawString("Shape",210,12) ;
       off2Gg.drawString("Area-sq in",260,12) ;
       off2Gg.drawString("Cam",327,12) ;
       off2Gg.drawString("AR",360,12) ;
 
       off2Gg.drawRect(10,382,170,15) ;
       off2Gg.drawString("Definitions and More Models",15,395) ;
       off2Gg.drawString("Click on 'Step 2.' button and set the Angle of Attack.",20,420) ;

//  Group 1
       if (group == 1) {
          off2Gg.setColor(Color.white) ;
          if (model == 1) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,15,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,15,190,30) ;
          off2Gg.drawString("1",2,35) ;
          exes[0] = 20 ;
          whys[0] = 30 ;
          exes[1] = 60 ;
          whys[1] = 30 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,20,20,20) ;
          off2Gg.drawString("6",80,35) ;
          off2Gg.drawString("1",176,35) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 2) {
            off2Gg.setColor(Color.white) ;
            off2Gg.fillRect(0,45,190,30) ;
            off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,45,190,30) ;
          off2Gg.drawString("2",2,65) ;
          exes[0] = 20 ;
          whys[0] = 60 ;
          exes[1] = 60 ;
          whys[1] = 60 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,55,35,15) ;
          off2Gg.drawString("6",85,68) ;
          off2Gg.drawString("4",176,65) ;

          off2Gg.setColor(Color.white) ;
          if (model == 3) {
            off2Gg.setColor(Color.white) ;
            off2Gg.fillRect(0,75,190,30) ;
            off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,75,190,30) ;
          off2Gg.drawString("3",2,95) ;
          exes[0] = 20 ;
          whys[0] = 90 ;
          exes[1] = 60 ;
          whys[1] = 90 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,84,60,12) ;
          off2Gg.drawString("6",100,95) ;
          off2Gg.drawString("6",176,95) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 4) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,105,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,105,190,30) ;
          off2Gg.drawString("4",2,125) ;
          exes[0] = 20 ;
          whys[0] = 121 ;
          exes[1] = 30 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 119 ;
          exes[1] = 40 ;
          whys[1] = 118 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 118 ;
          exes[1] = 50 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 119 ;
          exes[1] = 60 ;
          whys[1] = 121 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,110,20,20) ;
          off2Gg.drawString("6",80,125) ;
          off2Gg.drawString("1/12",139,125) ;
          off2Gg.drawString("1",176,125) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 5) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,135,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,135,190,30) ;
          off2Gg.drawString("5",2,155) ;
          exes[0] = 20 ;
          whys[0] = 150 ;
          exes[1] = 30 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 149 ;
          exes[1] = 40 ;
          whys[1] = 148 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 148 ;
          exes[1] = 50 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 149 ;
          exes[1] = 60 ;
          whys[1] = 150 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,140,20,20) ;
          off2Gg.drawString("6",80,155) ;
          off2Gg.drawString("1/16",139,155) ;
          off2Gg.drawString("1",176,155) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 6) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,165,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,165,190,30) ;
          off2Gg.drawString("6",2,185) ;
          exes[0] = 20 ;
          whys[0] = 180 ;
          exes[1] = 30 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 179 ;
          exes[1] = 40 ;
          whys[1] = 178 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 178 ;
          exes[1] = 50 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 179 ;
          exes[1] = 60 ;
          whys[1] = 180 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,170,20,20) ;
          off2Gg.drawString("6",80,185) ;
          off2Gg.drawString("1/20",139,185) ;
          off2Gg.drawString("1",176,185) ;

          off2Gg.setColor(Color.white) ;
          if (model == 7) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,195,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,195,190,30) ;
          off2Gg.drawString("7",2,215) ;
          exes[0] = 20 ;
          whys[0] = 211 ;
          exes[1] = 30 ;
          whys[1] = 209 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 209 ;
          exes[1] = 40 ;
          whys[1] = 208 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 208 ;
          exes[1] = 50 ;
          whys[1] = 209 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 209 ;
          exes[1] = 60 ;
          whys[1] = 211 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,204,60,12) ;
          off2Gg.drawString("6",100,215) ;
          off2Gg.drawString("1/12",139,215) ;
          off2Gg.drawString("6",176,215) ;

          off2Gg.setColor(Color.white) ;
          if (model == 8) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,225,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,225,190,30) ;
          off2Gg.drawString("8",2,245) ;
          exes[0] = 20 ;
          whys[0] = 241 ;
          exes[1] = 30 ;
          whys[1] = 240 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 240 ;
          exes[1] = 40 ;
          whys[1] = 239 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 239 ;
          exes[1] = 50 ;
          whys[1] = 240 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 240 ;
          exes[1] = 60 ;
          whys[1] = 241 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,234,60,12) ;
          off2Gg.drawString("6",100,245) ;
          off2Gg.drawString("1/16",139,245) ;
          off2Gg.drawString("6",176,245) ;

          off2Gg.setColor(Color.white) ;
          if (model == 9) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,255,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,255,190,30) ;
          off2Gg.drawString("9",2,275) ;
          exes[0] = 20 ;
          whys[0] = 271 ;
          exes[1] = 30 ;
          whys[1] = 270 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 270 ;
          exes[1] = 40 ;
          whys[1] = 269 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 269 ;
          exes[1] = 50 ;
          whys[1] = 270 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 270 ;
          exes[1] = 60 ;
          whys[1] = 271 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,264,60,12) ;
          off2Gg.drawString("6",100,275) ;
          off2Gg.drawString("1/20",139,275) ;
          off2Gg.drawString("6",176,275) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 10) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,285,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,285,190,30) ;
          off2Gg.drawString("10",2,305) ;
          exes[0] = 20 ;
          whys[0] = 301 ;
          exes[1] = 30 ;
          whys[1] = 300 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 300 ;
          exes[1] = 40 ;
          whys[1] = 299 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 299 ;
          exes[1] = 50 ;
          whys[1] = 298 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 298 ;
          exes[1] = 60 ;
          whys[1] = 301 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,294,60,12) ;
          off2Gg.drawString("6",100,305) ;
          off2Gg.drawString("1/12",139,305) ;
          off2Gg.drawString("6",176,305) ;

          off2Gg.setColor(Color.white) ;
          if (model == 11) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,315,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,315,190,30) ;
          off2Gg.drawString("11",2,335) ;
          exes[0] = 20 ;
          whys[0] = 330 ;
          exes[1] = 30 ;
          whys[1] = 329 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 329 ;
          exes[1] = 40 ;
          whys[1] = 329 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 329 ;
          exes[1] = 50 ;
          whys[1] = 328 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 328 ;
          exes[1] = 60 ;
          whys[1] = 330 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,324,60,12) ;
          off2Gg.drawString("6",100,335) ;
          off2Gg.drawString("1/16",139,335) ;
          off2Gg.drawString("6",176,335) ;

          off2Gg.setColor(Color.white) ;
          if (model == 12) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,345,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,345,190,30) ;
          off2Gg.drawString("12",2,365) ;
          exes[0] = 20 ;
          whys[0] = 360 ;
          exes[1] = 30 ;
          whys[1] = 359 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 359 ;
          exes[1] = 40 ;
          whys[1] = 359 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 359 ;
          exes[1] = 50 ;
          whys[1] = 358 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 358 ;
          exes[1] = 60 ;
          whys[1] = 360 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,354,60,12) ;
          off2Gg.drawString("6",100,365) ;
          off2Gg.drawString("1/20",139,365) ;
          off2Gg.drawString("6",176,365) ;

          off2Gg.setColor(Color.white) ;
          if (model == 15) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,15,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,15,190,30) ;
          off2Gg.drawString("15",192,35) ;
          exes[0] = 210 ;
          whys[0] = 31 ;
          exes[1] = 220 ;
          whys[1] = 30 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 30 ;
          exes[1] = 230 ;
          whys[1] = 29 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 29 ;
          exes[1] = 240 ;
          whys[1] = 28 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 28 ;
          exes[1] = 250 ;
          whys[1] = 31 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,20,20,20) ;
          off2Gg.drawString("6",270,35) ;
          off2Gg.drawString("1/12",329,35) ;
          off2Gg.drawString("1",366,35) ;
   
          off2Gg.setColor(Color.white) ;
          if (model == 16) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,45,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,45,190,30) ;
          off2Gg.drawString("16",192,65) ;
          exes[0] = 210 ;
          whys[0] = 60 ;
          exes[1] = 220 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 59 ;
          exes[1] = 230 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 59 ;
          exes[1] = 240 ;
          whys[1] = 58 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 58 ;
          exes[1] = 250 ;
          whys[1] = 60 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,50,20,20) ;
          off2Gg.drawString("6",270,65) ;
          off2Gg.drawString("1/16",329,65) ;
          off2Gg.drawString("1",366,65) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 17) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,75,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,75,190,30) ;
          off2Gg.drawString("17",192,95) ;
          exes[0] = 210 ;
          whys[0] = 90 ;
          exes[1] = 220 ;
          whys[1] = 89 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 89 ;
          exes[1] = 230 ;
          whys[1] = 89 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 89 ;
          exes[1] = 240 ;
          whys[1] = 88 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 88 ;
          exes[1] = 250 ;
          whys[1] = 90 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,80,20,20) ;
          off2Gg.drawString("6",270,95) ;
          off2Gg.drawString("1/20",329,95) ;
          off2Gg.drawString("1",366,95) ;
  
          off2Gg.setColor(Color.white) ;
          if (model == 18) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,105,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,105,190,30) ;
          off2Gg.drawString("18",192,125) ;
          exes[0] = 210 ;
          whys[0] = 120 ;
          exes[1] = 220 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 119 ;
          exes[1] = 230 ;
          whys[1] = 118 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 118 ;
          exes[1] = 240 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 119 ;
          exes[1] = 250 ;
          whys[1] = 120 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,115,35,15) ;
          off2Gg.drawString("6",275,128) ;
          off2Gg.drawString("1/20",329,125) ;
          off2Gg.drawString("4",366,125) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 19) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,135,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,135,190,30) ;
          off2Gg.drawString("19",192,155) ;
          exes[0] = 210 ;
          whys[0] = 150 ;
          exes[1] = 220 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 149 ;
          exes[1] = 230 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 149 ;
          exes[1] = 240 ;
          whys[1] = 148 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 148 ;
          exes[1] = 250 ;
          whys[1] = 150 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,145,35,15) ;
          off2Gg.drawString("6",275,158) ;
          off2Gg.drawString("1/20",329,155) ;
          off2Gg.drawString("4",366,155) ;
  
          off2Gg.setColor(Color.white) ;
          if (model == 20) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,165,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,165,190,30) ;
          off2Gg.drawString("20",192,185) ;
          exes[0] = 210 ;
          whys[0] = 180 ;
          exes[1] = 220 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 179 ;
          exes[1] = 230 ;
          whys[1] = 178 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 178 ;
          exes[1] = 240 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 179 ;
          exes[1] = 250 ;
          whys[1] = 180 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
       //  off2Gg.drawRect(260,170,20,20) ;
          exes[0] = 260 ;
          whys[0] = 175 ;
          exes[1] = 320 ;
          whys[1] = 175 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 316 ;
          whys[0] = 179 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 312 ;
          whys[1] = 182 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 304 ;
          whys[0] = 185 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 290 ;
          whys[1] = 187 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 276 ;
          whys[0] = 185 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 268 ;
          whys[1] = 182 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 264 ;
          whys[0] = 179 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 260 ;
          whys[1] = 175 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawString("6",290,185) ;
          off2Gg.drawString("4.7",361,185) ;
 
          off2Gg.setColor(Color.white) ;
          if (model == 21) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,195,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,195,190,30) ;
          off2Gg.drawString("21",192,215) ;
          exes[0] = 210 ;
          whys[0] = 210 ;
          exes[1] = 220 ;
          whys[1] = 209 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 209 ;
          exes[1] = 230 ;
          whys[1] = 209 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 209 ;
          exes[1] = 240 ;
          whys[1] = 208 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 208 ;
          exes[1] = 250 ;
          whys[1] = 210 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 260 ;
          whys[0] = 204 ;
          exes[1] = 320 ;
          whys[1] = 204 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 316 ;
          whys[0] = 208 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 312 ;
          whys[1] = 211 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 304 ;
          whys[0] = 214 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 290 ;
          whys[1] = 216 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 276 ;
          whys[0] = 214 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 268 ;
          whys[1] = 211 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 264 ;
          whys[0] = 208 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 260 ;
          whys[1] = 204 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          off2Gg.drawString("6",290,215) ;
          off2Gg.drawString("1/20",329,215) ;
          off2Gg.drawString("4.7",361,215) ;

          off2Gg.setColor(Color.white) ;
          if (model == 23) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,225,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,225,190,30) ;
          off2Gg.drawString("23",192,245) ;
          exes[0] = 210 ;
          whys[0] = 240 ;
          exes[1] = 220 ;
          whys[1] = 239 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 239 ;
          exes[1] = 230 ;
          whys[1] = 239 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 239 ;
          exes[1] = 240 ;
          whys[1] = 238 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 238 ;
          exes[1] = 250 ;
          whys[1] = 240 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,234,60,12) ;
          off2Gg.drawString("3",290,245) ;
          off2Gg.drawString("1/20",329,245) ;
          off2Gg.drawString("6.7",361,245) ;

          off2Gg.setColor(Color.white) ;
          if (model == 25) {
            off2Gg.setColor(Color.white) ;
            off2Gg.fillRect(190,255,190,30) ;
            off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,255,190,30) ;
          off2Gg.drawString("25",192,275) ;
          exes[0] = 210 ;
          whys[0] = 270 ;
          exes[1] = 220 ;
          whys[1] = 269 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 269 ;
          exes[1] = 230 ;
          whys[1] = 269 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 269 ;
          exes[1] = 240 ;
          whys[1] = 268 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 268 ;
          exes[1] = 250 ;
          whys[1] = 270 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 260 ;
          whys[0] = 264 ;
          exes[1] = 320 ;
          whys[1] = 264 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 318 ;
          whys[0] = 267 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 313 ;
          whys[1] = 271 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 310 ;
          whys[0] = 274 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 305 ;
          whys[1] = 276 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 275 ;
          whys[0] = 276 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 270 ;
          whys[1] = 274 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 267 ;
          whys[0] = 271 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 262 ;
          whys[1] = 267 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 260 ;
          whys[0] = 264 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          off2Gg.drawString("4",290,275) ;
          off2Gg.drawString("1/20",329,275) ;
          off2Gg.drawString("8",366,275) ;

          off2Gg.setColor(Color.white) ;
          if (model == 30) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,285,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,285,190,30) ;
          off2Gg.drawString("30",192,305) ;
          exes[0] = 210 ;
          whys[0] = 300;
          exes[1] = 220 ;
          whys[1] = 299 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 299 ;
          exes[1] = 230 ;
          whys[1] = 298 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 298 ;
          exes[1] = 240 ;
          whys[1] = 297 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 297 ;
          exes[1] = 250 ;
          whys[1] = 302 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,294,60,12) ;
          off2Gg.drawString("5",290,305) ;
          off2Gg.drawString("1/6",329,305) ;
          off2Gg.drawString("4.1",361,305) ;

          off2Gg.setColor(Color.white) ;
          if (model == 31) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,315,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,315,190,30) ;
          off2Gg.drawString("31",192,335) ;
          exes[0] = 210 ;
          whys[0] = 331 ;
          exes[1] = 220 ;
          whys[1] = 330 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 330 ;
          exes[1] = 230 ;
          whys[1] = 329 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 329 ;
          exes[1] = 240 ;
          whys[1] = 328 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 328 ;
          exes[1] = 250 ;
          whys[1] = 331 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[1] = 320 ;
          whys[1] = 330 ;
          exes[0] = 316 ;
          whys[0] = 328 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 312 ;
          whys[1] = 327 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 304 ;
          whys[0] = 325 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 290 ;
          whys[1] = 324 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 276 ;
          whys[0] = 325 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 268 ;
          whys[1] = 327 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 264 ;
          whys[0] = 328 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 260 ;
          whys[1] = 330 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 264 ;
          whys[0] = 332 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 268 ;
          whys[1] = 333 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 276 ;
          whys[0] = 335 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 290 ;
          whys[1] = 336 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 304 ;
          whys[0] = 335 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 312 ;
          whys[1] = 333 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 316 ;
          whys[0] = 332 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 320 ;
          whys[1] = 330 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawString("8",290,335) ;
          off2Gg.drawString("1/12",329,335) ;
          off2Gg.drawString("4.6",361,335) ;

          off2Gg.setColor(Color.white) ;
          if (model == 35) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,345,190,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,345,190,30) ;
          off2Gg.drawString("35",192,365) ;
          exes[0] = 210 ;
          whys[0] = 360 ;
          exes[1] = 220 ;
          whys[1] = 359 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 359 ;
          exes[1] = 230 ;
          whys[1] = 359 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 359 ;
          exes[1] = 240 ;
          whys[1] = 358 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 358 ;
          exes[1] = 250 ;
          whys[1] = 360 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,354,60,12) ;
          off2Gg.drawString("6",290,365) ;
          off2Gg.drawString("1/10",329,365) ;
          off2Gg.drawString("6",366,365) ;
  // 51
          off2Gg.setColor(Color.white) ;
          if (model == 51) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,375,200,30) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,375,200,30) ;
          off2Gg.drawString("51",192,395) ;
          exes[0] = 210 ;
          whys[0] = 390 ;
          exes[1] = 220 ;
          whys[1] = 389 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 389 ;
          exes[1] = 230 ;
          whys[1] = 389 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 389 ;
          exes[1] = 240 ;
          whys[1] = 388 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 388 ;
          exes[1] = 250 ;
          whys[1] = 390 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,384,60,12) ;
          off2Gg.drawString("6.25",285,395) ;
          off2Gg.drawString("1/11",329,395) ;
          off2Gg.drawString("4",366,395) ;
       }
 // Group 2
       if (group == -1) {
          off2Gg.setColor(Color.white) ;
          if (model == 24) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,15,190,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,15,190,60) ;
          off2Gg.drawString("24",2,50) ;
          exes[0] = 20 ;
          whys[0] = 30 ;
          exes[1] = 30 ;
          whys[1] = 29 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 29 ;
          exes[1] = 40 ;
          whys[1] = 29 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 29 ;
          exes[1] = 50 ;
          whys[1] = 28 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 28 ;
          exes[1] = 60 ;
          whys[1] = 30 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 20 ;
          whys[0] = 60 ;
          exes[1] = 30 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 59 ;
          exes[1] = 40 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 59 ;
          exes[1] = 50 ;
          whys[1] = 58 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 58 ;
          exes[1] = 60 ;
          whys[1] = 60 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,40,60,12) ;
          off2Gg.drawString("6",100,51) ;

          off2Gg.drawString("1/20",139,35) ;
          off2Gg.drawString("6.7",171,35) ;
          off2Gg.drawString("Gap",145,50) ;
          off2Gg.drawString("5/8",145,65) ;
          off2Gg.drawString("Foils 22 & 23",65,70) ;
   // 27
          off2Gg.setColor(Color.white) ;
          if (model == 27) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,75,190,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,75,190,60) ;
          off2Gg.drawString("27",2,110) ;
          exes[0] = 20 ;
          whys[0] = 90 ;
          exes[1] = 30 ;
          whys[1] = 89 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 89 ;
          exes[1] = 40 ;
          whys[1] = 89 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 89 ;
          exes[1] = 50 ;
          whys[1] = 88 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 88 ;
          exes[1] = 60 ;
          whys[1] = 90 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 20 ;
          whys[0] = 105 ;
          exes[1] = 30 ;
          whys[1] = 104 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 104 ;
          exes[1] = 40 ;
          whys[1] = 104 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 104 ;
          exes[1] = 50 ;
          whys[1] = 103 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 103 ;
          exes[1] = 60 ;
          whys[1] = 105 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 20 ;
          whys[0] = 120 ;
          exes[1] = 30 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 119 ;
          exes[1] = 40 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 119 ;
          exes[1] = 50 ;
          whys[1] = 118 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 118 ;
          exes[1] = 60 ;
          whys[1] = 120 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(70,100,60,12) ;
          off2Gg.drawString("9",100,111) ;

          off2Gg.drawString("1/20",139,95) ;
          off2Gg.drawString("6.7",171,95) ;
          off2Gg.drawString("Gap",145,110) ;
          off2Gg.drawString("5/8",145,125) ;
          off2Gg.drawString("Foils 22, 23, 26",60,130) ;
  // 33
          off2Gg.setColor(Color.white) ;
          if (model == 33) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(0,135,190,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(0,135,190,60) ;
          off2Gg.drawString("33",2,170) ;
          exes[0] = 20 ;
          whys[0] = 150 ;
          exes[1] = 30 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 149 ;
          exes[1] = 40 ;
          whys[1] = 149 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 149 ;
          exes[1] = 50 ;
          whys[1] = 148 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 148 ;
          exes[1] = 60 ;
          whys[1] = 150 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 20 ;
          whys[0] = 180 ;
          exes[1] = 30 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 30 ;
          whys[0] = 179 ;
          exes[1] = 40 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 40 ;
          whys[0] = 179 ;
          exes[1] = 50 ;
          whys[1] = 178 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 50 ;
          whys[0] = 178 ;
          exes[1] = 60 ;
          whys[1] = 180 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 70 ;
          whys[0] = 160 ;
          exes[1] = 130 ;
          whys[1] = 160 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 128 ;
          whys[0] = 163 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 123 ;
          whys[1] = 167 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 120 ;
          whys[0] = 170 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 115 ;
          whys[1] = 172 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 85 ;
          whys[0] = 172 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 80 ;
          whys[1] = 170 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 77 ;
          whys[0] = 167 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          exes[1] = 72 ;
          whys[1] = 163 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 70 ;
          whys[0] = 160 ;
          off2Gg.drawLine(exes[1],whys[1],exes[0],whys[0]) ;
          off2Gg.drawString("8",100,170) ;

          off2Gg.drawString("1/20",139,155) ;
          off2Gg.drawString("8",174,155) ;
          off2Gg.drawString("Gap",145,170) ;
          off2Gg.drawString("11/16",145,185) ;
          off2Gg.drawString("Foils 25 & 32",65,190) ;
   //41
          off2Gg.setColor(Color.white) ;
          if (model == 40) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,15,200,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,15,198,60) ;
          off2Gg.drawString("40",192,55) ;
          exes[0] = 210 ;
          whys[0] = 35 ;
          exes[1] = 220 ;
          whys[1] = 34 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 34 ;
          exes[1] = 230 ;
          whys[1] = 34 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 34 ;
          exes[1] = 240 ;
          whys[1] = 33 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 33 ;
          exes[1] = 250 ;
          whys[1] = 35 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 210 ;
          whys[0] = 60 ;
          exes[1] = 220 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 59 ;
          exes[1] = 230 ;
          whys[1] = 59 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 59 ;
          exes[1] = 240 ;
          whys[1] = 58 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 58 ;
          exes[1] = 250 ;
          whys[1] = 60 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,40,60,12) ;
          off2Gg.drawString("12",290,50) ;

          off2Gg.drawString("1/20",329,38) ;
          off2Gg.drawString("4",364,38) ;
          off2Gg.drawString("Gap",335,53) ;
          off2Gg.drawString("11/16",335,68) ;
          off2Gg.drawString("Foils 18 & 19",255,73) ;
   //41
          off2Gg.setColor(Color.white) ;
          if (model == 41) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,75,200,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,75,198,60) ;
          off2Gg.drawString("41",192,115) ;
          exes[0] = 210 ;
          whys[0] = 95 ;
          exes[1] = 220 ;
          whys[1] = 94 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 94 ;
          exes[1] = 230 ;
          whys[1] = 94 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 94 ;
          exes[1] = 240 ;
          whys[1] = 93 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 93 ;
          exes[1] = 250 ;
          whys[1] = 95 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 210 ;
          whys[0] = 120 ;
          exes[1] = 220 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 119 ;
          exes[1] = 230 ;
          whys[1] = 119 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 119 ;
          exes[1] = 240 ;
          whys[1] = 118 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 118 ;
          exes[1] = 250 ;
          whys[1] = 120 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,100,60,12) ;
          off2Gg.drawString("12",290,110) ;

          off2Gg.drawString("1/20",329,98) ;
          off2Gg.drawString("4",364,98) ;
          off2Gg.drawString("Gap",335,113) ;
          off2Gg.drawString("18/16",335,128) ;
          off2Gg.drawString("Foils 18 & 19",255,133) ;
 // 43
          off2Gg.setColor(Color.white) ;
          if (model == 43) {
             off2Gg.setColor(Color.white) ;
             off2Gg.fillRect(190,135,200,60) ;
             off2Gg.setColor(Color.blue) ;
          }
          off2Gg.drawRect(190,135,200,60) ;
          off2Gg.drawString("43",192,175) ;
          exes[0] = 210 ;
          whys[0] = 155 ;
          exes[1] = 220 ;
          whys[1] = 154 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 154 ;
          exes[1] = 230 ;
          whys[1] = 154 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 154 ;
          exes[1] = 240 ;
          whys[1] = 153 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 153 ;
          exes[1] = 250 ;
          whys[1] = 155 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;

          exes[0] = 210 ;
          whys[0] = 180 ;
          exes[1] = 220 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 220 ;
          whys[0] = 179 ;
          exes[1] = 230 ;
          whys[1] = 179 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 230 ;
          whys[0] = 179 ;
          exes[1] = 240 ;
          whys[1] = 178 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          exes[0] = 240 ;
          whys[0] = 178 ;
          exes[1] = 250 ;
          whys[1] = 180 ;
          off2Gg.drawLine(exes[0],whys[0],exes[1],whys[1]) ;
          off2Gg.drawRect(260,160,60,12) ;
          off2Gg.drawString("12",290,170) ;

          off2Gg.drawString("1/20",329,158) ;
          off2Gg.drawString("4",364,158) ;
          off2Gg.drawString("Gap",335,173) ;
          off2Gg.drawString("1/4",335,188) ;
          off2Gg.drawString("Foils 18 & 19",255,193) ;

          off2Gg.drawString("AR - Aspect Ratio -  span / chord",50,220) ;
          off2Gg.drawString("           span  is distance from tip to tip",50,235) ;
          off2Gg.drawString("           chord is distance from front to back",50,250) ;
          off2Gg.drawString("Cam - Camber or Curvature - height / chord",50,270) ;
          off2Gg.drawString("           1/12 is more curved than 1/20",50,285) ;
          off2Gg.drawString("Gap - Space between wings ",50,305) ;
       }
     } // stepper = 1

       g.drawImage(offImg2,0,0,this) ;
    }
  } //Slvpnl 
}
